<?php

namespace Inside\Content\Jobs;

use CURLFile;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Jobs\Job;
use Intervention\Image\ImageManager;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

/**
 * Class GenerateStylizedImage
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class GenerateStylizedImage extends Job
{
    /**
     * Image path
     */
    protected string $image;

    public function __construct(
        string $image,
        protected ImageStyles $style
    ) {
        // For security reason remove first / & .
        $this->image = ltrim(ltrim($image, '.'), '/');
    }

    /**
     * Generate an optimized/stylized image from our original source
     */
    public function handle(): void
    {
        /** @var array $info */
        $info = pathinfo($this->image);
        if (isset($info['extension']) && strtolower($info['extension']) === 'gif') {
            return;
        }

        Log::debug('[GenerateStylizedImage] handle ['.$this->image.'] <'.$this->style->title.'>');
        // Generate image
        $width = $this->style->width;
        $height = $this->style->height;
        $optimisation = (bool) $this->style->optimisation;
        $optimisationRatio = ! $optimisation ? 100 : $this->style->optimisation_ratio;
        $resizable = $this->style->resizable;
        $styleDirectory = $this->style->title;
        $imageDirectory = 'main';
        $disk = Storage::disk('local');

        if (! $disk->exists($this->image)) {
            // Original image not found !
            Log::warning("[GenerateStylizedImage] can't find original file [".$this->image.']');

            return;
        }

        if ($width == 0) {
            $width = null;
        }

        if ($height == 0) {
            $height = null;
        }

        // we don't filter on height because if height = 0 then it's an image paragraph
        if ($width === null) {
            return;
        }

        // Load original image
        try {
            $manager = new ImageManager(['driver' => 'gd']);
            $image = $manager->make($disk->path($this->image));
        } catch (Exception) {
            try {
                $manager = new ImageManager(['driver' => 'imagick']);
                $image = $manager->make($disk->path($this->image));
            } catch (Exception) {
                Log::error('cant read image '.$this->image);

                return;
            }
        }

        // we generate a webp for the main image too if it doesn't exist
        if (! $disk->exists('styles/'.$imageDirectory)) {
            $mainImage = clone $image;
            $mainImage = $mainImage->orientate();
            $mainImage = $mainImage->encode('webp', 100);
            $mainWebpPath = 'styles/'.$imageDirectory.'/'.$info['dirname'].'/'.$info['filename'].'-'.($info['extension'] ?? '').'.webp';
            $disk->put($mainWebpPath, $mainImage, 'private');
            $mainImage->destroy();
        }

        // Auto oriente image
        $image = $image->orientate();

        if (! is_null($height)) {
            // Auto oriente & Resize image
            $method = $resizable ? 'resize' : 'fit';
            $image->{$method}(
                $width,
                $height,
                function ($constraint) {
                    $constraint->aspectRatio();
                }
            )->resizeCanvas($width, $height);
        } else {
            $image->resize(
                $width,
                $image->getHeight(),
                function ($constraint) {
                    $constraint->aspectRatio();
                    $constraint->upsize();
                }
            );
        }

        if (! env('USE_RESMUSH', false)) {
            // Optimize
            $webp = clone $image;
            $image = $image->encode(null, $optimisationRatio);
            $webpImage = $webp->encode('webp', 100);

            // Save stylized image
            $disk->put('styles/'.$styleDirectory.'/'.$this->image, $image, 'private');
            $webpPath = 'styles/'.$styleDirectory.'/'.$info['dirname'].'/'.$info['filename'].'-'.($info['extension'] ?? '').'.webp';
            $disk->put($webpPath, $webpImage, 'private');
            $image->destroy();
            $webpImage->destroy();
        } else {
            // Optimize resized image
            $resmushImage = $this->getOptimizedByResmush($disk->path('styles/'.$styleDirectory.'/'.$this->image));
            if ($resmushImage !== null) {
                $disk->put('styles/'.$styleDirectory.'/'.$this->image, $resmushImage, 'private');
            }
        }
        Log::debug(
            '[GenerateStylizedImage] image <'.$this->image.'> generated for style <'.$this->style->title.'>'
        );
    }

    /**
     * Optimize image using Resmush.it
     */
    protected function getOptimizedByResmush(string $imagePath): ?string
    {
        $guesser = MimeTypeGuesser::getInstance();

        $file = new CURLFile($imagePath, $guesser->guess($imagePath), basename($imagePath));

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'http://api.resmush.it/?qlty=80');
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
        curl_setopt(
            $ch,
            CURLOPT_POSTFIELDS,
            [
                'files' => $file,
            ]
        );
        $result = curl_exec($ch);
        if (curl_errno($ch)) {
            $result = curl_error($ch);
        }
        curl_close($ch);

        $result = json_decode((string) $result);
        if ($result) {
            if (! isset($result->error)) {
                $content = file_get_contents($result->dest);
                if ($content !== false) {
                    return $content;
                }
            } else {
                Log::error('[getOptimizedByResmush] '.$result->error);
            }
        }

        return null;
    }
}
