<?php

namespace Inside\Content\Http\Middlewares;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Content\Events\UserMentionedInCommentEvent;
use Inside\Content\Facades\ContentHelper;
use Inside\Http\Middleware\TransformResponseAndRequest;

/**
 * Class NotificationRequestRole
 */
class UserMentionRequest extends TransformResponseAndRequest
{
    protected array $paths = [
        'POST' => [
            '#^api/v1/content/([^/]+)$#',
        ],
    ];

    protected ?array $mentionedUsers = [];

    protected ?string $contentType;

    protected function patchRequest(Request $request): Request
    {
        $this->mentionedUsers = ContentHelper::getAndRemoveQuery($request, 'mentioned_users');

        return $request;
    }

    /**
     * Patch response
     */
    protected function patch(Request $request, mixed $response, array $data = []): mixed
    {
        if (! $response instanceof JsonResponse) {
            return $response;
        }
        $this->contentType = $data[1];

        if (! in_array($this->contentType, ['comments', 'groups_posts'])) {
            return $response;
        }

        if (count($data) === 2) {
            return $this->patchOnStore($request, $response);
        }

        return $response;
    }

    /**
     * distpach event on content creation
     *
     * @param Request $request
     * @param JsonResponse $response
     * @return mixed
     */
    protected function patchOnStore(Request $request, JsonResponse $response)
    {
        if (empty($this->mentionedUsers)) {
            return $response;
        }

        $responseData = (array) json_decode_response($response);

        if (! isset($responseData['uuid'])) {
            return $response;
        }

        foreach ($this->mentionedUsers as $mentionedUser) {
            event(new UserMentionedInCommentEvent((string) $this->contentType, $responseData['uuid'], $mentionedUser));
        }

        return $response;
    }
}
