<?php

namespace Inside\Content\Http\Middlewares;

use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Events\ContentSavedWithImages;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Content;
use Inside\Content\Models\WysiwygImage;
use Inside\Database\Eloquent\Builder;
use Inside\Http\Middleware\TransformResponseAndRequest;
use Inside\Permission\Scopes\AllowedScope;
use Inside\Support\Str;

class SaveWysiwygImages extends TransformResponseAndRequest
{
    protected array $paths = [
        'POST' => [
            '#^api/v1/content/([^/]+)$#',
        ],
        'PUT' => [
            '#^api/v1/content/([^/]+)/([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})$#',
        ],
        'DELETE' => [
            '#^api/v1/content/([^/]+)$#',
        ],
    ];

    protected ?string $contentType;

    protected array $wysiwygImages = [];

    protected string $type;

    protected ?string $id = null;

    protected ?string $langcode;

    protected string $callback;

    protected function patchRequest(Request $request): Request
    {
        $this->callback = Str::camel('path_'.$request->getMethod());

        $matches = [];
        if (preg_match(Arr::first($this->paths[$request->getMethod()]), $request->path(), $matches) > 0) {
            $this->type = $matches[1];
            if (isset($matches[2])) {
                $this->id = $matches[2];
            }
        }
        $this->langcode = $request->get('langcode');
        $this->wysiwygImages = ContentHelper::getAndRemoveQuery($request, 'wysiwyg_images') ?? [];

        return $request;
    }

    /**
     * Patch response
     */
    protected function patch(Request $request, mixed $response, array $data = []): mixed
    {
        $responseData = (array) json_decode_response($response);

        if (! isset($responseData['uuid']) || empty($this->type)) {
            return $response;
        }

        /**
         * @var WysiwygImageService $wysiwygImageService
         */
        $wysiwygImageService = app()->make(WysiwygImageService::class);

        /** @var Builder $query */
        $query = call_user_func(type_to_class($this->type).'::query');

        /** @var Content $content */
        $content = $query->withoutGlobalScope(AllowedScope::class)->find($this->id ?? $responseData['uuid']);

        if ($responseData['uuid'] && $this->langcode && $content->langcode !== $this->langcode) {
            $originalContent = $content;
            $class = type_to_class($this->type);
            $content = call_user_func($class.'::find', $responseData['uuid']);
            $originalContentImages = [];
            collect($this->wysiwygImages)->each(function ($image) use (
                $class,
                $originalContent,
                $content,
                &$originalContentImages,
                $wysiwygImageService
            ) {
                $wysiwygImage = WysiwygImage::where([
                    'imageable_id' => $originalContent->uuid,
                    'imageable_type' => $class,
                    'filename' => basename($image),
                ])->first();

                if (! is_null($wysiwygImage)) {
                    $originalContentImages[$image] = true;
                    $wysiwygImageService->translate($originalContent, $content, $wysiwygImage);
                }
            });

            $this->wysiwygImages = collect($this->wysiwygImages)
                ->reject(fn ($image) => collect($originalContentImages)->has($image))
                ->values()
                ->all();
        }

        if (! empty($this->wysiwygImages)) {
            $wysiwygImageService->store($content, $this->wysiwygImages);
        }

        ContentSavedWithImages::dispatch($content);

        return $response;
    }
}
