<?php

declare(strict_types=1);

namespace Inside\Content\Http\Controllers\Revision;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Arr;
use Inside\Content\Contracts\RevisionService;
use Inside\Content\Exceptions\RevisionCanNotRestored;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Content;
use Laravel\Lumen\Routing\Controller as BaseController;

final class Restore extends BaseController
{
    use UseRevision;

    /**
     * @throws RevisionCanNotRestored
     */
    public function __invoke(
        RevisionService $service,
        string $type,
        string $uuid,
        int $version
    ): JsonResponse {
        $content = ContentHelper::find($type, $uuid);
        if (! $content instanceof Content
            || ! $service->isEnabled($content)
            || ! $service->canBeRestored($content, $version)) {
            abort(404);
        }

        $restoredRevision = $this->getRevision($content, $version);
        if (is_null($restoredRevision) || $content->current_revision->id === $restoredRevision->id) {
            throw RevisionCanNotRestored::for($content, $restoredRevision);
        }

        return response()->json([
            'success' => $service->restoreFromRevision($content, $restoredRevision),
            'slug' => Arr::first($content->fresh()?->slug) ?? $content->uuid,
        ]);
    }
}
