<?php

declare(strict_types=1);

namespace Inside\Content\Http\Controllers\Revision;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\RevisionService;
use Inside\Content\Exceptions\InvalidRevision;
use Inside\Content\Exceptions\RevisionCanNotBeFound;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Http\Resources\UserResource;
use Inside\Content\Models\Content;
use Inside\Content\Models\Revision;
use Laravel\Lumen\Routing\Controller as BaseController;

final class Index extends BaseController
{
    use UseRevision;

    /**
     * @throws RevisionCanNotBeFound|InvalidRevision
     */
    public function __invoke(
        RevisionService $service,
        string $type,
        string $uuid,
        string $version,
        string|null $baseVersion = null
    ): JsonResponse {
        $content = ContentHelper::find($type, $uuid);
        if (! $content instanceof Content || ! $service->isEnabled($content)) {
            abort(404);
        }

        $baseRevision = null;
        if ($baseVersion === 'current') {
            $baseRevision = $content->current_revision;
            $baseVersion = $content->current_revision?->version;
        } elseif (is_numeric($baseVersion)) {
            $baseVersion = intval($baseVersion);
            $baseRevision = $this->getRevision($content, $baseVersion);
        }

        if ($version === 'current') {
            $version = $content->current_revision?->version;
        } elseif ($version === 'previous') {
            $version = $content->current_revision?->version - 1;
            if ($version < 1) {
                $version = null;
            }
        } elseif (is_numeric($version)) {
            $version = intval($version);
        } else {
            $version = null;
        }

        if (is_null($version)) {
            Log::debug(__('[Revision::Index] failed to get version'));
            throw RevisionCanNotBeFound::for($content, $version);
        }
        $revision = $this->getRevision($content, $version);

        if (is_null($revision) || $baseRevision?->id === $revision->id) {
            Log::debug(__('[Revision::Index] trying to get no revision or comparing two same revision'));
            throw InvalidRevision::for($content);
        }
        $data = [];
        if (! is_null($baseRevision)) {
            $data['base'] = $this->getResourceForRevision($service, $content, $baseRevision, $baseVersion);
        }

        return response()->json(array_merge($data, [
            'revision' => $this->getResourceForRevision($service, $content, $revision, $version),
            'meta' => $content->revisions->map(fn (Revision $revision) => [
                'version' => $revision->version,
                'date' => $revision->created_at->timestamp,
            ]),
        ]));
    }

    protected function getResourceForRevision(
        RevisionService $service,
        Content $content,
        Revision $revision,
        int $version
    ): array {
        return [
            'version' => $version,
            'can_be_restored' => $service->canBeRestored($content, $version),
            'date' => $revision->created_at->timestamp,
            'user' => $revision->user instanceof User ? UserResource::make($revision->user) : null,
            'data' => $service->transformRevisionToFrontView($revision),
        ];
    }
}
