<?php

namespace Inside\Content\Http\Controllers\Resource;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Inside\Authentication\Models\User;
use Inside\Kernel\Filesystem\Facades\MimeTypeValidator;
use Inside\Permission\Exceptions\AuthorizationException;
use Laravel\Lumen\Routing\Controller;
use Pion\Laravel\ChunkUpload\Exceptions\UploadFailedException;
use Pion\Laravel\ChunkUpload\Exceptions\UploadMissingFileException;
use Pion\Laravel\ChunkUpload\Handler\HandlerFactory;
use Pion\Laravel\ChunkUpload\Receiver\FileReceiver;

final class Upload extends Controller
{
    /**
     * Upload a new file.
     *
     * @throws UploadMissingFileException
     * @throws UploadFailedException*@throws Exception
     * @throws Exception
     */
    public function __invoke(Request $request): JsonResponse
    {
        /** @var ?User $user */
        $user = Auth::user();
        $type = Str::slug($request->get('type', 'system'));

        if (is_null($user)) {
            throw AuthorizationException::create('upload', $type);
        }

        /**
         * @var UploadedFile $file
         */
        $file = $request->file('file');

        if (! MimeTypeValidator::validate($file)) {
            throw new Exception('File mime type is not allowed');
        }
        $receiver = new FileReceiver('file', $request, HandlerFactory::classFromRequest($request));

        if ($receiver->isUploaded() === false) {
            throw new UploadMissingFileException();
        }

        if (is_bool($save = $receiver->receive())) {
            throw new Exception('Reseiver is in wrong state');
        }

        if ($save->isFinished()) {
            $file = $save->getFile();

            // Security check !
            if (! $file->isValid()) {
                throw ValidationException::withMessages([Lang::get('content.upload.file_is_not_valid')]);
            }
            if ($this->shouldBlockPhpUpload($file)) {
                throw ValidationException::withMessages([Lang::get('content.upload.php_file_is_not_allowed')]);
            }
            $time = time();

            $extension = $file->getClientOriginalExtension();
            $originalName = $file->getClientOriginalName();
            $baseName = pathinfo($originalName, PATHINFO_FILENAME);
            $fileName = $baseName.'.'.$extension;

            Storage::disk('local')->putFileAs('chunks/'.$time, $file, $fileName);

            return response()->json([
                'path' => 'chunks/'.$time.'/'.$fileName,
            ]);
        }

        $handler = $save->handler();

        return response()->json([
            'done' => $handler->getPercentageDone(),
        ]);
    }

    /**
     * File should be blocked because it is detected as a php file !
     */
    protected function shouldBlockPhpUpload(UploadedFile $file): bool
    {
        $phpExtensions = [
            'php',
            'php3',
            'php4',
            'php5',
            'php7',
            'php8',
            'pht',
            'phtml',
        ];

        return in_array(
            trim(strtolower($file->getClientOriginalExtension())),
            $phpExtensions
        );
    }

    protected function sanitizeFilename(UploadedFile $file): string
    {
        $extension = $file->getClientOriginalExtension();
        $originalName = $file->getClientOriginalName();

        if ($extension === 'wkdownload') {
            $extension = pathinfo($originalName, PATHINFO_EXTENSION);
        }

        $baseName = pathinfo($originalName, PATHINFO_FILENAME);
        $baseName = str_replace(' ', '-', $baseName);
        $baseName = (string) preg_replace('/[^A-Za-z0-9\-\_]/', '', $baseName);
        $baseName = (string) preg_replace('/-+/', '-', $baseName);

        return $baseName.'.'.$extension;
    }
}
