<?php

namespace Inside\Content\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Models\WysiwygImage;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Slug\Models\Slug;
use Laravel\Lumen\Routing\Controller;

class PublicPagesController extends Controller
{
    public function __construct(
        private ContentTransformer $transformer
    ) {
    }

    public function __invoke(WysiwygImageService $service, string $slug): JsonResponse
    {
        $uuid = Slug::whereSlug($slug)->whereType('inside_content_public_pages')->firstOrFail()->uuid;

        $query = call_user_func(type_to_class('public_pages').'::withoutGlobalScopes')->where('uuid', $uuid);

        $data = $this->transformer->transformCollection(
            collection: $query->get(),
            fields: ['uuid', 'title', 'content', 'content_type']
        );

        $files = collect($data['data'][0]['content'])
            ->map(fn (array $section) => match ($section['section_type']) {
                'file', 'video' => $section['file']['path'],
                'image' => $section['image']['main'],
                'text' => str($section['body'])->matchAll('/src="(.*?)"/'),
                'text_with_image' => [$section['image']['main'], str($section['body'])->matchAll('/src="(.*?)"/')],
                'slider' => collect($section['content'])->pluck('image.main')->toArray(),
                default => null,
            })
            ->flatten()
            ->mapWithKeys(fn ($url) => match (true) {
                str($url)->contains('wysiwyg') => [$url => $service->load(str($url)->after('/wysiwyg/images/'))],
                str($url)->contains('video') => [$url => (string) str($url)->after('/video/')],
                str($url)->contains('slide') => [$url => (string) str($url)->after('/slide/')],
                str($url)->contains('text_with_image') => [$url => (string) str($url)->after('/text_with_image/')],
                default => [$url => (string) str($url)->after('/file/')->after('/image/')],
            })
            ->map(function ($path) {
                if ($path instanceof WysiwygImage) {
                    $path = sprintf('wysiwyg/images/public_pages/%s/%s', $path->imageable_id, $path->internal);
                }

                $public = 'public/'.$path;

                if (Storage::disk('local')->exists($path) && ! Storage::disk('local')->exists($public)) {
                    Storage::disk('local')->copy($path, $public);
                }

                return (string) str($public)->after('/')->start('/images/')->replace('/', '\/');
            })
            ->mapWithKeys(fn ($value, $key) => [(string) str($key)->replace('/', '\/') => $value])
            ->toArray();

        return response()->json(
            json_decode(strtr((string) json_encode($data), $files), true)
        );
    }
}
