<?php

namespace Inside\Content\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Content\Services\OpenWeatherService;

class InsideWeatherController
{
    protected string $apiKey;

    public function __construct(private OpenWeatherService $openWeatherService)
    {
    }

    public function getWeather(string $uuid): JsonResponse
    {
        $errors = [];

        $contentType = config('openweathermap.source_content_type');
        if (! InsideSchema::hasModel($contentType)) {
            $errors = [
                'success' => false,
                'error' => sprintf('content type %s not found', $contentType),
            ];
        }

        /**
         * @var Content $content
         */
        $content = type_to_class($contentType)::where('uuid', $uuid)->first();
        if ($content === null) { // @phpstan-ignore-line
            $errors = [
                'success' => false,
                'error' => sprintf('content %s not found with uuid %s', $contentType, $uuid),
            ];
        }

        if (! empty($errors)) {
            return response()->json($errors, 404);
        }

        $weatherData = $this->openWeatherService
            ->setContent($content)
            ->getWeather();

        $status = empty($weatherData) ? 204 : 200;

        return response()->json($weatherData, $status);
    }

    public function getWeatherByCities(Request $request): JsonResponse
    {
        $errors = [];

        $validated = $request->validate([
            'query' => 'required|array|min:1',
            'query.*' => 'required|array|size:2',
            'query.*.0' => 'required|numeric|between:-90,90',
            'query.*.1' => 'required|numeric|between:-180,180',
        ]);

        if (empty($validated)) {
            $errors = [
                'success' => false,
                'error' => 'Invalid request',
            ];
        }

        $coords = $validated['query'];
        $weatherData = collect($coords)
            ->map(fn ($coord) => $this->openWeatherService->getWeatherByCoord($coord[0], $coord[1]))
            ->filter()
            ->values()
            ->toArray();

        if (empty($weatherData)) {
            $errors = [
                'success' => false,
                'error' => 'No weather data found',
            ];
        }

        if (! empty($errors)) {
            return response()->json($errors, 404);
        }

        return response()->json($weatherData, 200);
    }

    public function getLocationsForCity(Request $request): JsonResponse
    {
        $errors = [];

        $validated = $request->validate([
            'query' => 'required|string|max:255',
        ]);

        if (empty($validated)) {
            $errors = [
                'success' => false,
                'error' => 'Invalid request',
            ];
        }

        $city = $validated['query'];

        $weatherData = $this->openWeatherService->getLocationsForCity($city);

        if (empty($weatherData)) {
            $errors = [
                'success' => false,
                'error' => sprintf('No weather data found for city: %s', $city),
            ];
        }

        if (! empty($errors)) {
            return response()->json($errors, 404);
        }

        return response()->json($weatherData, 200);
    }

    public function getLocationsForLatlong(Request $request): JsonResponse
    {
        $errors = [];

        $validated = $request->validate([
            'query' => 'required|array|min:1',
            'query.*' => 'required|array|size:2',
            'query.*.0' => 'required|numeric|between:-90,90',
            'query.*.1' => 'required|numeric|between:-180,180',
        ]);

        if (empty($validated)) {
            $errors = [
                'success' => false,
                'error' => 'Invalid request',
            ];
        }

        $latlong = $validated['query'];

        $weatherData = [];
        try {
            $weatherData = collect($latlong)->flatMap(
                fn ($coord) => $this->openWeatherService->getLocationsForLatlong($coord[0], $coord[1])
            );
        } catch (\Exception $exception) {
            $errors = [
                'success' => false,
                'error' => $exception->getMessage(),
            ];
        }

        if (! empty($errors)) {
            return response()->json($errors, 404);
        }

        return response()->json($weatherData, 200);
    }
}
