<?php

namespace Inside\Content\Http\Controllers;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Support\Str;
use Laravel\Lumen\Routing\Controller;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

/**
 * Inside file controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class FileController extends Controller
{
    /**
     * Upload assets
     *
     * @OA\Schema(
     *    schema="UploadAssetResult",
     *    type="object",
     *           @OA\Property(
     *                property="path",
     *                description="La chemin absolu vers le fichier téléchargé",
     *                type="string",
     *                example="https://sid2.intranet-inside.com/assets/system/monfichier.png"
     *           )
     *  )
     *
     * @OA\Post(
     *      path="/content/asset",
     *      operationId="assetUpload",
     *      tags={"Contenu"},
     *      summary="Permet de charger des fichiers assets dans Inside.",
     *     @OA\RequestBody(
     *     required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *            @OA\Schema(
     *                 @OA\Property(
     *                     property="type",
     *                     description="Le type d'asset à envoyer ( par défault **system** ) ",
     *                     example="settings",
     *                     type="string",
     *                     default="system"
     *                 ),
     *                 @OA\Property(
     *                     property="asset",
     *                     description="Le fichier a sauvegarder",
     *                     type="string",
     *                     format="binary"
     *                 ),
     *                 required={"asset"}
     *           )
     *        )
     *     ),
     *     @OA\Response(
     *          response=200,
     *          description="Le chemin absolu pour accéder au fichier téléchargé ",
     *          @OA\JsonContent(ref="#/components/schemas/UploadAssetResult")
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     */
    public function upload(Request $request): JsonResponse
    {
        // Check rights
        /** @var User|null $me */
        $me = Auth::user();
        if (! $me) {
            throw AuthorizationException::create('upload', 'settings');
        }

        // Reset Mime guesser, drupal mess everything
        MimeTypeGuesser::reset();

        // Validate asset
        $maxSize = UploadedFile::getMaxFilesize();

        $extensionsFromSettings = collect([
            'file_extensions'  => setting('contentManagement', 'file_extensions', '[]'),
            'img_extensions'   => setting('contentManagement', 'img_extensions', '[]'),
            'video_extensions' => setting('contentManagement', 'video_extensions', '[]'),
        ])
            ->map(fn ($ext) => collect(json_decode($ext, true)))
            ->flatten()
            ->filter()
            ->unique()
            ->values()
            ->implode(',');

        $this->validate(
            $request,
            [
                'type'  => 'string|alpha_dash',
                'asset' => 'required|file|mimes:'.$extensionsFromSettings.'|max:'.$maxSize,
            ]
        );

        $type = Str::slug($request->get('type', 'system'));

        $disk = Storage::disk('assets');

        if (! $request->has('asset')) {
            abort(500);
        }
        /** @var UploadedFile $asset */
        $asset = $request->file('asset');

        $file = $disk->putFileAs($type, $asset, $this->hashName($asset));

        return response()->json([
            'url' => $disk->url($file),
            'name' => $asset->getClientOriginalName(),
            'size' => $asset->getSize(),
            'type' => $asset->getMimeType(),
        ]);
    }

    /**
     * Delete an existing assets
     *
     *
     * @OA\Schema(
     *    schema="DeleteAssetResult",
     *    type="object",
     *           @OA\Property(
     *                property="successful",
     *                description="Indique si le fichier a bien été effacé",
     *                type="boolean",
     *                example=true
     *           )
     *  )
     *
     * @OA\Delete(
     *      path="/content/asset",
     *      operationId="assetDestroy",
     *      tags={"Contenu"},
     *      summary="Permet d'effacer un fichier asset dans Inside.",
     *     @OA\Parameter(
     *         name="type",
     *         in="query",
     *         description="Le type d'asset à envoyer ( par défault **system** ) ",
     *         required=false,
     *         example="settings",
     *         @OA\Schema(
     *             type="string",
     *             default="system"
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="name",
     *         in="query",
     *         description="Le nom du fichier",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\Response(
     *          response=200,
     *          description="Si l'opération a réussi ou non ",
     *          @OA\JsonContent(ref="#/components/schemas/UploadAssetResult")
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     * @throws AuthenticationException
     */
    public function destroy(Request $request): JsonResponse
    {
        // Check rights
        /** @var User|null $me */
        $me = Auth::user();
        if (! $me) {
            throw new AuthenticationException();
        }

        // Validate asset
        $this->validate(
            $request,
            [
                'type' => 'string|alpha_dash',
                'name' => 'required|string',
            ]
        );

        $type = Str::slug($request->get('type', 'system'));

        return response()->json([
            'successful' => Storage::disk('assets')->delete($type.'/'.$request->get('name')),
        ]);
    }

    /**
     * hash Name
     */
    protected function hashName(UploadedFile $file, string $path = null): string
    {
        if ($path) {
            $path = rtrim($path, '/').'/';
        }

        $hash = Str::random(40);

        return $path.$hash.'.'.$file->guessExtension();
    }
}
