<?php

namespace Inside\Content\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Exceptions\AuthenticationException;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Calendar;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Users;
use Inside\Reaction\Facades\Reactions;
use Inside\Reaction\Models\Reaction;
use Inside\Support\Str;
use InvalidArgumentException;
use Laravel\Lumen\Routing\Controller;
use League\Csv\Writer;

class EventsController extends Controller
{
    private ?User $currentUser;

    protected array $types;

    protected const CSV_DOWNLOAD_FOLDER = 'csv/download/events';

    public function __construct()
    {
        /** @var User $currentUser */
        $currentUser = Auth::user();
        $this->currentUser = $currentUser;
        $this->types = config('events.type');
    }

    /**
     * @throws AuthenticationException
     */
    protected function handleErrors(Request $request): void
    {
        if (! $request->has('uuid')) {
            throw new InvalidArgumentException('Missing uuid');
        }

        if (! $this->currentUser) {
            throw new AuthenticationException();
        }

        foreach ($this->types as $type) {
            if (! Schema::hasModel($type)) {
                throw ModelSchemaNotFoundException::named($type);
            }
        }

        $type = $request->get('type', 'events');

        if (! in_array($type, $this->types)) {
            throw new InvalidArgumentException('Content type '.$type.' is not allowed');
        }
    }

    /**
     * @throws Exception
     */
    public function register(Request $request): string
    {
        $this->handleErrors($request);
        $type = $request->get('type', 'events');
        $uuid = $request->get('uuid');
        $class = type_to_class($type);
        $eventReactions = DB::table('inside_reactions')
            ->where('type', 'register')
            ->where('reactionable_type', $class)
            ->where('reactionable_uuid', $uuid)
            ->count();

        $requestedEvent = call_user_func($class.'::findOrFail', $uuid);
        $translateContents = $class::where('uuid_host', $requestedEvent->uuid_host)
            ->get();

        if (isset($requestedEvent->participants_number) && $requestedEvent->participants_number > 0 && $requestedEvent->participants_number <= $eventReactions) {
            throw new Exception('limit of participants already reached');
        }

        $translateContents->each(
            fn ($event) => Reactions::create([
                'user_uuid' => $this->currentUser?->uuid,
                'reactionable_uuid' => $event->uuid,
                'reactionable_type' => $class,
                'type' => 'register',
                'langcode' => $event->langcode,
            ])
        );

        return Calendar::getEventCalendarExport('ics', $requestedEvent);
    }

    /**
     * @throws AuthenticationException
     */
    public function unregister(Request $request): JsonResponse
    {
        $this->handleErrors($request);
        $type = $request->get('type', 'events');
        $uuid = $request->get('uuid');
        $class = type_to_class($type);

        $requestedEvent = call_user_func($class.'::findOrFail', $uuid);
        $translateContents = $class::where('uuid_host', $requestedEvent->uuid_host)->get();

        $translateContents->each(
            fn ($event) => Reaction::query()
                ->where('type', 'register')
                ->where('user_uuid', $this->currentUser?->uuid)
                ->where('reactionable_type', $class)
                ->where('reactionable_uuid', $event->uuid)
                ->first()
                ?->delete()
        );

        return response()->json([
            'success' => true,
        ]);
    }

    public function export(Request $request): JsonResponse
    {
        $this->handleErrors($request);
        $type = $request->get('type', 'events');
        $uuid = $request->get('uuid');
        $participantsUuids = array_filter(explode(',', $request->get('participants', '')));
        $class = type_to_class($type);
        $data = call_user_func($class.'::findOrFail', $uuid);

        $bonusExportFields = config('events.export_fields', []);

        $participants = DB::table('inside_reactions')
            ->where('type', 'register')
            ->where('reactionable_type', $class)
            ->where('reactionable_uuid', $uuid)
            ->when($participantsUuids, fn ($query) => $query->whereIn('user_uuid', $participantsUuids))
            ->pluck('user_uuid');

        $headers = [
            Lang::get('export.titles.firstname'),
            Lang::get('export.titles.lastname'),
            Lang::get('export.titles.email'),
        ];

        foreach ($bonusExportFields as $field) {
            if (! Lang::has('export.titles.'.$field)) {
                $headers[] = $field;
                continue;
            }
            $headers[] = Lang::get('export.titles.'.$field);
        }

        if (! Storage::disk('protected')->exists(self::CSV_DOWNLOAD_FOLDER)) {
            Storage::disk('protected')->makeDirectory(self::CSV_DOWNLOAD_FOLDER);
        }

        $slug = Str::slug($data->title, '_').'_'.time();
        $filePath = self::CSV_DOWNLOAD_FOLDER.'/'.$slug.'.csv';
        $csv = Writer::createFromPath(Storage::disk('protected')->path($filePath), 'w+')
            ->setDelimiter(';');
        $csv->setOutputBOM(Writer::BOM_UTF8);
        $csv->insertOne($headers);

        foreach ($participants as $participant) {
            $user = Users::find($participant);
            $line = [
                $user->firstname,
                $user->lastname,
                $user->email,
            ];

            foreach ($bonusExportFields as $field) {
                $line[] = $user->{$field} ?? '';
            }

            $csv->insertOne($line);
        }
        $disableProtectedFiles = config('app.disable_protected_files', false);
        $path = $disableProtectedFiles ? env('APP_URL').'/'.config('app.app_storage_path').'/protected/'.$filePath :
            Storage::disk('protected')->url($filePath);

        return response()->json([
            'url' => $path,
        ]);
    }
}
