<?php

namespace Inside\Content\Http\Controllers;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\ContentList as ContentListContract;
use Inside\Content\Contracts\Transformer;
use Inside\Content\Facades\ContentCache;
use Inside\Content\Facades\ContentCleaner;
use Inside\Content\Facades\ContentExporter;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\ContentImporter;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContent;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Permission\Facades\Permission;
use Inside\Slug\Models\Slug;
use Inside\Slug\Services\SlugService;
use Inside\Validation\ValidateRequests;
use Laravel\Lumen\Routing\Controller;
use OpenApi\Annotations as OA;
use Symfony\Component\HttpFoundation\Response;

/**
 * Inside content controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 *
 * @OA\Tag(name="Contenu",description="
# Introduction

Il s'agit du service principe d'inside qui permet de récupérer, de créer, de modifier ou d'effacer un contenu.

")
 */
class ContentController extends Controller
{
    use ValidateRequests;

    /**
     * List contents
     *
     *
     * @OA\Schema(
     *           schema="ContentList",
     *           type="object"
     *  )
     *
     * @OA\Get(
     *      path="/content",
     *      operationId="contentList",
     *      tags={"Contenu"},
     *      summary="Permet de récupérer le contenu d'un ou plusieurs types.",
     *     @OA\Parameter(
     *         name="types",
     *         in="path",
     *         description="Le ou les types de contenus désirés. Si plusieurs types sont requis, il faut indiquer
    les types séparés par une virgule. Le type de données est le nom machien du contenu.",
     *         required=true,
     *         @OA\Schema(
     *             type="string"
     *         ),
     *         example="news,events"
     *     ),
     *
     *     @OA\Response(
     *          response=200,
     *          description="Le contenu",
     *          @OA\JsonContent(ref="#/components/schemas/ContentList")
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     *  )
     */
    public function list(string $types, Request $request, ContentListContract $list): JsonResponse
    {
        if ($cached = ContentCache::has($types, $request->toArray())) {
            return response()->json($cached);
        }

        $data = $list($types, $request);

        if (! $request->has('nocache') || ! $request->get('nocache')) {
            ContentCache::store($types, $request->toArray(), $data);
        }

        return response()->json($data);
    }

    /**
     * @throws Exception
     */
    public function export(string $type, Request $request): JsonResponse
    {
        // Assuming that to be allowed to export you must be allowed to create a $type content
        if ($type !== 'users' && ! Permission::allowed('create', $type)) {
            throw AuthorizationException::create('export', $type);
        }

        // Check driver
        $driver = $request->get('driver');

        if (! ContentExporter::hasDriver($driver)) {
            throw new Exception('Driver ['.$driver.'] is not a valid export driver');
        }

        $configName = $request->get('config');
        $configs = config('imports');
        $supported = array_keys($configs);

        $config = [
            'driver' => $driver ?? null,
            'content_type' => $type,
            'fields' => $request->get('fields'),
            'input' => $request->all(),
            'config' => null,
        ];

        if (in_array($configName, $supported)) {
            if ($configs[$configName]['content_type'] != $type) {
                throw new Exception(
                    'Type ['.$type.'] does not match content type from config ['.$configName.']'
                );
            }
            if (is_null($config['driver'])) {
                // Override config driver
                $config['driver'] = $configs[$configName]['format']['driver'];
            }
            $config['fields'] = null;
            $config['config'] = $configName;
        }

        return response()->json(['success' => ContentExporter::export($config['driver'], $config)]);
    }

    /**
     * @throws Exception
     */
    public function import(string $type, Request $request): JsonResponse
    {
        // Assuming that to be allowed to export you must be allowed to create a $type content
        if (! Permission::allowed('create', $type)) {
            throw AuthorizationException::create('import', $type);
        }

        // Check driver
        $driver = $request->get('driver');

        // Get config name
        $configName = $request->get('config');
        if ($configName === null) {
            throw new Exception('You must provide a config name');
        }

        // Load configs
        $configs = config('imports');
        $supported = array_keys($configs);

        // Check required parameters
        if (! in_array($configName, $supported)) {
            throw new Exception('You must provide a valid config name ['.$configName.']');
        }
        $config = $configs[$configName];
        if ($config['content_type'] != $type) {
            throw new Exception('Type ['.$type.'] does not match content type from config ['.$configName.']');
        }
        if ($driver) {
            // Override config driver
            $config['driver'] = $driver;
        }

        if (! $config['driver']) {
            throw new Exception('You must set a driver to export');
        }

        if (! ContentImporter::hasDriver($config['driver'])) {
            throw new Exception('Driver ['.$config['driver'].'] is not a valid export driver');
        }

        return response()->json(['success' => ContentImporter::import($config)]);
    }

    /**
     * Create a new content.
     * @throws Exception
     */
    public function create(Request $request, Transformer $transformer, string $type): JsonResponse
    {
        if (! in_array($type, ['comments']) && ! Permission::allowed('create', $type)) {
            throw AuthorizationException::create('create', $type);
        }
        $data = $request->except('fields');

        // L'auteur est toujours unique et correspond à une colonne de la table,
        // mais le front envoi un tableau ...
        if (isset($data['author']) && is_array($data['author'])) {
            $data['author'] = Arr::first($data['author']);
            $data['author_id'] = $data['author'];
        }

        if (isset($data['update_author']) && is_array($data['update_author'])) {
            $data['update_author'] = Arr::first($data['update_author']);
        }

        $data = ContentHelper::addMissings($type, $data);
        $data = ContentHelper::castAttributes($type, $data);
        if ($type == 'tv_screens') {
            $data['screen_id'] = type_to_class($type)::max('screen_id') + 1;
        }
        if ($type == 'comments') {  // Special comments case
            // Cleanup front stuff
            unset($data['uid']);
            unset($data['authors']);
            $rules = ContentHelper::makeCommentRules($data['bundle']);
        } else {
            /** @var User $user */
            $user = Auth::user();
            $rules = ContentHelper::makeRules($type, null, $user->permission->hasAnyRole('super_administrator'));
        }
        $data = $this->validateData($data, $rules, ['password.regex' => Lang::get('validation.password.regex')], ContentHelper::getAttributeNames($type));

        $fields = [];
        $bridge = new BridgeContent();
        $contentUuid = $bridge->contentInsert($type, $data);

        if (is_null($contentUuid)) {
            abort(
                500,
                (string) json_encode(
                    [
                        'error' => 'bridge did not correctly work ',
                        'type' => $type,
                    ]
                )
            );
        }

        if ($request->has('fields')) {
            $fields = $request->get('fields');
        }
        try {
            $query = call_user_func(type_to_class($type).'::withoutGlobalScopes');

            $content = $query->findOrFail($contentUuid);
        } catch (Exception $e) {
            Log::error(json_encode($e) ?: "Error: {$e->getMessage()} ({$e->getFile()}:{$e->getLine()})");

            return response()->json(
                [
                    'error' => $e->getMessage(),
                ],
                400
            );
        }

        return response()->json($transformer->transform($content, $fields));
    }

    /**
     * Update a content by its id.
     */
    public function update(
        Request $request,
        Transformer $transformer,
        string $type,
        string $id
    ): JsonResponse {
        if (! Permission::allowed('update', $type, $id)) {
            throw AuthorizationException::create('update', $type);
        }
        $data = $request->except('fields');
        /** @var User $user */
        $user = Auth::user();

        // L'auteur est tjrs unique et correspond à une colonne de la table
        // mais le front envoi un tableau ...
        if (isset($data['author']) && is_array($data['author'])) {
            $data['author'] = Arr::first($data['author']);
        }
        if (isset($data['update_author']) && is_array($data['update_author'])) {
            $data['update_author'] = Arr::first($data['update_author']);
        }

        $data = ContentHelper::addMissings($type, $data, false);
        $data = ContentHelper::castAttributes($type, $data);
        $rules = ContentHelper::makeRules(
            type: $type,
            uuid: $id,
            force: $user->permission->hasAnyRole('super_administrator') || ($type === 'users' && Permission::userCanAccessBackofficeEntry('user', $user))
        );

        $data = $this->validateData(
            $data,
            $rules,
            [],
            ContentHelper::getAttributeNames($type)
        );

        $fields = [];

        $class = type_to_class($type);
        $bridge = new BridgeContent();
        $content = call_user_func($class.'::find', $id);

        if ($content->langcode === $data['langcode']) {
            $data['uuid'] = $id;
        } else {
            if (array_key_exists('uuid', $data)) {
                unset($data['uuid']);
            }
            $data['uuid_host'] = $content->uuid_host;
        }

        $reservedFields = ['uuid_host', 'uuid', 'type', 'bundle', 'langcode'];
        foreach ($data as $field => &$value) {
            if (
                ! in_array($field, $reservedFields)
                && ! Schema::hasField($type, $field)
                && ! in_array($field, ['author', 'author_id', 'update_author', 'published_at'])
            ) {
                unset($data[$field]);
                continue;
            }

            if (! is_array($value)) {
                continue;
            }

            foreach ($value as $key => $val) {
                if (empty($val)) {
                    unset($value[$key]);
                }
            }
        }

        try {
            $contentUuid = $bridge->contentUpdate($type, $data);
        } catch (Exception $e) {
            Log::error(json_encode($e) ?: "Error: {$e->getMessage()} ({$e->getFile()}:{$e->getLine()})");

            return response()->json(
                [
                    'error' => $e->getMessage(),
                ],
                400
            );
        }
        if ($request->has('fields')) {
            $fields = $request->get('fields');

            if (is_string($fields)) {
                $fields = json_decode($fields, true) ?? [];
            }
        }

        try {
            $query = call_user_func($class.'::query');

            $content = $query->findOrFail($contentUuid);
        } catch (Exception $e) {
            Log::error(json_encode($e) ?: "Error: {$e->getMessage()} ({$e->getFile()}:{$e->getLine()})");

            return response()->json(
                [
                    'error' => $e->getMessage(),
                ],
                400
            );
        }

        // get translated versions infos
        if ($type !== 'users') {
            array_push($fields, 'translations');
        }

        return response()->json($transformer->transform($content, $fields));
    }

    /**
     * Delete a content by its id.
     */
    public function delete(Request $request, string $type, string $id): JsonResponse
    {
        if (! Permission::allowed('delete', $type, $id)) {
            throw AuthorizationException::create('delete', $type);
        }

        $content = call_user_func(type_to_class($type).'::findOrFail', $id);

        ContentCleaner::validate($content);

        ContentCleaner::forceChildrenDeletion($content, $request->query->getBoolean('force_children_deletion'));

        $bridge = new BridgeContent();
        $service = new SlugService();

        return response()->json(
            [
                'status' => $bridge->contentDelete($type, $id),
                'parent_slug' => $service->getParentSlug($type, $id),
                'uuid' => $id,
            ]
        );
    }

    public function deleteMultiple(Request $request): JsonResponse
    {
        $data = $request->get('contents');
        $type = '';
        $deleteFails = [];
        $contents = [];

        foreach ($data as $type => $uuids) {
            foreach ($uuids as $uuid) {
                if (! Permission::allowed('delete', $type, $uuid)) {
                    $deleteFails[] = [
                        'type' => $type,
                        'uuid' => $uuid,
                    ];
                } else {
                    $contents[] = call_user_func(type_to_class($type).'::query')->where('uuid', $uuid)->first();
                }
            }
        }

        $forceChildrenDeletion = $request->query->getBoolean('force_children_deletion');

        $bridge = new BridgeContent();
        $service = new SlugService();
        $uuids = [
            'succeed' => [],
            'failed' => [],
        ];

        foreach ($contents as $content) {
            ContentCleaner::validate($content);
            ContentCleaner::forceChildrenDeletion($content, $forceChildrenDeletion);

            $service->getParentSlug(class_to_type($content), $content->uuid);

            $status = $bridge->contentDelete($type, $content->uuid);
            if ($status) {
                array_push($uuids['succeed'], $content->uuid);
            } else {
                array_push($uuids['failed'], $content->uuid);
            }
        }

        if ($deleteFails) {
            throw AuthorizationException::create('delete', $type);
        }

        return response()->json([
            'uuids' => $uuids,
        ]);
    }

    public function getSlugInfos(Request $request, string $slug): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        /** @var string $filters */
        $filters = $request->query('filters') ?? '';
        $filters = json_decode($filters, true);
        $langcode = $filters['langcode'] ?? null;

        $slug = Slug::where('slug', $slug)
            ->when($langcode !== null, fn (Builder $query) => $query->where('langcode', $langcode))
            ->first();

        if (! $slug) {
            return new JsonResponse(['message' => 'Slug model not found !'], Response::HTTP_NOT_FOUND);
        }

        $model = call_user_func(table_to_class($slug->type).'::find', $slug->uuid);
        if (! $model) {
            return new JsonResponse(['message' => 'Model related to the given slug not found !'], Response::HTTP_NOT_FOUND);
        }

        $contentsInfo = [];
        $languages = list_languages();
        foreach ($languages as $langcode) {
            $content = $model->getTranslationIfExists($langcode);
            if ($content->langcode !== $langcode) {
                $contentsInfo[$langcode] = [];
                continue;
            }

            $contentsInfo[$langcode] = [
                'uuid' => $content->uuid,
                'content_type' => $content->content_type,
                'slug' => $content->slug,
                'langcode' => $content->langcode,
            ];
        }

        return new JsonResponse(['data' => $contentsInfo]);
    }
}
