<?php

declare(strict_types=1);

namespace Inside\Content\Contracts;

use Closure;
use Exception;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use stdClass;

interface SchemaService
{
    /**
     * Get models
     *
     * @return array<string, array>|null
     */
    public function models(): ?array;

    /**
     * Get old deprecated system drupal schema
     *
     * @return array<string, array>
     */
    public function schema(): array;

    /**
     * @return array<string, array<int, string>>
     */
    public function getFieldNamesThatReferenceType(string $needleType): array;

    /**
     * Do we have schema for this type
     *
     * NOTE: Schema is different from model, schema has reversed field, special fields etc...
     */
    public function hasSchemaInformation(string $type): bool;

    /**
     * Get schema information for type
     *
     * NOTE: Schema is different from model, schema has reversed field, special fields etc...
     * @return array<string, array>|null
     */
    public function getSchemaInformation(string $type): ?array;

    /**
     * Get all types
     *
     * @return array<int, string>
     */
    public function getTypes(?Closure $callback = null): array;

    /**
     * Get Content types name
     *
     * @return array<int, string>
     */
    public function getContentTypes(Closure $callback = null): array;

    /**
     * Get Section types name
     *
     * @return array<int, string>
     */
    public function getSectionTypes(Closure $callback = null): array;

    /**
     * Does this type is a content type or a section type,
     */
    public function isModel(string $type): bool;

    /**
     * Does this type is a content type ,
     */
    public function isContentType(string $type): bool;

    /**
     * Does this type is a section type ,
     */
    public function isSectionType(string $type): bool;

    /**
     * Get Full Model information as an object
     */
    public function getModelInformation(string $modelName): stdClass;

    /**
     * Get Model options
     *
     * @param  string  $modelName  model Name
     *
     * @return array<string, mixed>
     * @throws ModelSchemaNotFoundException
     */
    public function getModelOptions(string $modelName): array;

    /**
     * Set a model option and save in database
     */
    public function setModelOption(string $modelName, string $optionName, mixed $value): void;

    /**
     * Check if $modelName exists
     */
    public function hasModel(string $modelName): bool;

    /**
     * Check if $modelName exists
     */
    public function hasSectionType(string $modelName): bool;

    /**
     * Check if $modelName exists
     */
    public function hasContentType(string $modelName): bool;

    /**
     * Get all models with a given field
     *
     * @return array<int, string>
     */
    public function getModelsWithField(string $fieldName): array;

    /**
     * Get fields for a model
     *
     * @return array<int, string>
     */
    public function getFieldListing(string $modelName, ?Closure $callback = null): array;

    /**
     * @return array<int, string>
     */
    public function getRelations(string $type): array;

    /**
     * @return array<int, string>
     */
    public function getSortedDisplayedFieldListing(string $modelName): array;

    /**
     * Only get field of a certain type
     *
     * @return array<int, string>
     */
    public function getFieldListingOfType(string $modelName, string|array $types): array;

    /**
     * Only get field of a certain type for any models
     *
     * @return array<string, array<int, string>>
     */
    public function getAllFieldsListingOfType(string $type): array;

    /**
     * Only get field of a certain type for any models
     *
     * @return array<string, array<int, string>>
     */
    public function getAllFieldsListingOfWidget(string $widget): array;

    /**
     * get Displayed field listing for $modelName
     *
     * @return array<int, string>
     */
    public function getDisplayedFieldListing(string $modelName): array;

    /**
     * Get Field options
     * @return array<string, mixed>
     * @throws FieldSchemaNotFoundException
     * @throws ModelSchemaNotFoundException
     */
    public function getFieldOptions(string $modelName, string $fieldName): array;

    /**
     * Does a model has Field ?
     */
    public function hasField(string $modelName, string $fieldName): bool;

    /**
     * Does our model have at least one field of type $type
     */
    public function hasFieldOfType(string $modelName, string $type): bool;

    /**
     * Does our model modelName has fieldNames
     *
     * @throws Exception
     */
    public function hasFields(string $modelName, array $fieldNames): bool;

    /**
     * Force reload from database
     */
    public function refresh(): void;

    /**
     * Does this field is a system field ?
     */
    public function isSystemField(string $fieldName): bool;

    /**
     * Get base name from real field name
     */
    public function getOriginalFieldName(string $fieldName): string;

    /**
     * @return array<string, mixed>
     */
    public function formatFieldToLegacyFront(string $fieldName, array $fieldOptions): array;
}
