<?php

namespace Inside\Content\Console;

use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside;
use Inside\Console\Command;

/**
 * Class SyncPivotsCommand
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class SyncPivotsCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:pivots:sync';

    /**
     * @var string
     */
    protected $signature = 'inside:pivots:sync
                            {--S|silent=false}';

    /**
     * @var string
     */
    protected $description = 'Resync all pivots';

    /**
     * Handle index command
     *
     * @return void
     * @throws Exception
     */
    public function handle(): void
    {
        /** @var string|null $silent */
        $silent = $this->option('silent');
        $silent = $silent === null;
        $now = Carbon::now();
        $models = Inside\Content\Models\Model::all();
        $pivots = DB::table('inside_pivots')->get();

        $silent or $this->info(Inside::getCommandInfo());

        $silent or $this->info(
            'Starting to sync at ['.$now->toFormattedDateString().' '.$now->toTimeString().']'
        );

        foreach ($pivots as $pivot) {
            $parentType = null;
            $relatedType = null;
            $parentLangcode = null;
            $relatedLangcode = null;

            foreach ($models as $model) {
                if (! $parentType && $parent = call_user_func($model->class.'::find', $pivot->parent_uuid)) {
                    $parentType = $model->class;
                    $parentLangcode = $parent->langcode;
                }
                if (! $relatedType && $related = call_user_func($model->class.'::find', $pivot->related_uuid)) {
                    $relatedType = $model->class;
                    $relatedLangcode = $related->langcode;
                }
            }

            if (! $parentType || ! $relatedType) {
                DB::table('inside_pivots')->where('id', $pivot->id)->delete();
                continue;
            }

            DB::table('inside_pivots')->where('id', $pivot->id)->update(
                [
                    'parent_type'      => $parentType,
                    'parent_langcode'  => $parentLangcode,
                    'related_type'     => $relatedType,
                    'related_langcode' => $relatedLangcode,
                    'related_field'    => Str::snake(
                        str_replace('Inside\Content\Models\Contents\\', '', $relatedType)
                    ),
                ]
            );
        }

        $silent or $this->line('');
        $silent or $this->info('Synchronization done, started ['.$now->diffForHumans().']');
    }
}
