<?php

namespace Inside\Content\Console;

use Illuminate\Console\Command;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;

class SetContentTypesAsTranslatableCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:content:set-content-types-as-translatable';

    /**
     * @var string
     */
    protected $signature = 'inside:content:set-content-types-as-translatable';

    /**
     * @var string
     */
    protected $description = 'Sets all inside content types and their fields as translatable.';

    /**
     * @throws ModelSchemaNotFoundException
     */
    public function handle(): void
    {
        $bridge = new BridgeContentType();
        $contents = collect(Schema::getContentTypes())->diff(['users', 'admin_menus', 'comments', 'footer_menus', 'main_menus', 'shortcuts_menus', 'tools_menus']);

        $contents->each(function (string $contentType) use ($bridge) {
            if (! Schema::hasModel($contentType)) {
                return;
            }

            $this->getOutput()->title("[+] Type de contenu: $contentType");

            if (! data_get(Schema::getModelOptions($contentType), 'translatable', false)) {
                $bridge->contentTypeUpdate($contentType, ['translatable' => true]);
                $this->getOutput()->success('    Le type de contenu est maintenant translatable');
            } else {
                $this->getOutput()->text('    Le type de contenu était déjà translatable');
            }

            if (! data_get(Schema::getFieldOptions($contentType, 'langcode'), 'displayed', false)) {
                $bridge->changeFieldFormOptions($contentType, 'langcode', true, 'group_advanced_options');
                $this->getOutput()->success('    Le champ langcode est maintenant affiché');
            } else {
                $this->getOutput()->text('    Le champ langcode était déjà affiché');
            }

            $fields = collect(Schema::getFieldListing($contentType))->diff(array_merge(['langcode', 'authors', 'status', 'title', 'created_at', 'updated_at'], collect(Schema::getRelations($contentType))->toArray()));

            $fields->each(function (string $field) use ($bridge, $contentType) {
                if (! data_get(Schema::getFieldOptions($contentType, $field), 'translatable', false)) {
                    $bridge->contentTypeUpdateField($contentType, $field, ['translatable' => true]);
                    $this->getOutput()->success("    Le champ $field est maintenant translatable");
                } else {
                    $this->getOutput()->text("    Le champ $field était déjà translatable");
                }
            });
        });
    }
}
