<?php

namespace Inside\Content\Console;

use Exception;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schema;
use Inside\Console\Command;
use Inside\Content\Facades\DynamicClass;
use Inside\Facades\Package;
use Inside\Permission\Exodus\Actions\GenerateMissing\GenerateMissingBackofficeSections;
use Inside\Permission\Exodus\Actions\GenerateMissing\GenerateMissingCapabilities;
use Inside\Permission\Exodus\Actions\GenerateMissing\GenerateMissingCategorizablePrivileges;
use Inside\Permission\Exodus\Actions\GenerateMissing\GenerateMissingContentPrivileges;
use Inside\Permission\Exodus\Actions\GenerateMissing\GenerateMissingContentTypePrivileges;
use Inside\Permission\Exodus\Actions\GenerateMissing\GenerateMissingMenuPrivileges;
use Inside\Permission\Exodus\Actions\RefreshViewTables\RefreshCategorizableContentIndexesViewTable;
use Inside\Permission\Exodus\Actions\RefreshViewTables\RefreshCategorizableContentTranslationIndexesViewTable;
use Inside\Permission\Exodus\Actions\RefreshViewTables\RefreshContentTranslationIndexesViewTable;
use Inside\Permission\Exodus\Actions\RefreshViewTables\RefreshMenuIndexesViewTable;
use Inside\Permission\Exodus\Actions\RemoveUnnecessaries\RemoveUnnecessaryBackofficeSections;
use Inside\Permission\Exodus\Actions\RemoveUnnecessaries\RemoveUnnecessaryCapabilities;
use Inside\Permission\Exodus\Actions\RemoveUnnecessaries\RemoveUnnecessaryCategorizablePrivileges;
use Inside\Permission\Exodus\Actions\RemoveUnnecessaries\RemoveUnnecessaryContentPrivileges;
use Inside\Permission\Exodus\Actions\RemoveUnnecessaries\RemoveUnnecessaryContentTypePrivileges;
use Inside\Permission\Exodus\Actions\RemoveUnnecessaries\RemoveUnnecessaryMenuPrivileges;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Services\PermissionService;

/**
 * Rebuild dynamic model command.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class RebuildModelsCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'models:rebuild';

    /**
     * @var string
     */
    protected $description = 'Rebuild dynamic classes';

    /**
     * @throws Exception
     */
    public function handle(): int
    {
        $this->call('cache:clear');
        DynamicClass::rebuild();

        $this->info('Dynamics classes have been regenerated');

        if (! PermissionService::isSystemV2Initialized()) {
            return Command::SUCCESS;
        }

        /**
         * Refresh View Tables
         */
        $this->warn('Regenerating indexes views...');
        (new RefreshMenuIndexesViewTable())->execute();
        (new RefreshCategorizableContentIndexesViewTable())->execute();
        (new RefreshCategorizableContentTranslationIndexesViewTable())->execute();
        (new RefreshContentTranslationIndexesViewTable())->execute();
        $this->info('Indexes views have been regenerated');

        /**
         * Refresh Capability / Privileges
         */
        $privileges = [
            'capabilities' => [
                'remove' => new RemoveUnnecessaryCapabilities(),
                'generate' => new GenerateMissingCapabilities(),
            ],
            'backoffice sections' => [
                'remove' => new RemoveUnnecessaryBackofficeSections(),
                'generate' => new GenerateMissingBackofficeSections(),
            ],
            'menu privileges' => [
                'remove' => new RemoveUnnecessaryMenuPrivileges(),
                'generate' => new GenerateMissingMenuPrivileges(),
            ],
            'content type privileges' => [
                'remove' => new RemoveUnnecessaryContentTypePrivileges(),
                'generate' => new GenerateMissingContentTypePrivileges(),
            ],
            'categorizable privileges' => [
                'remove' => new RemoveUnnecessaryCategorizablePrivileges(),
                'generate' => new GenerateMissingCategorizablePrivileges(),
            ],
            'content privileges' => [
                'remove' => new RemoveUnnecessaryContentPrivileges(),
                'generate' => new GenerateMissingContentPrivileges(),
            ],
        ];

        foreach ($privileges as $title => $actions) {
            $this->info("Remove potential unnecessary $title...");
            $unnecessaries = $actions['remove']->execute();

            if ($unnecessaries) {
                $this->warn("$unnecessaries unnecessaries $title have been removed");
            }

            $this->info("Regenerating potential missing $title...");
            $missing = $actions['generate']->execute();

            if ($missing) {
                $this->warn("$missing missing $title have been regenerated");
            }
        }

        $this->call('inside:stats:tables');

        $this->info('Verification of the stats table has ended !');

        if (Package::has('inside-newsletters')) {
            $this->call('inside:stats:newsletters:tables');
        }

        return Command::SUCCESS;
    }
}
