<?php

declare(strict_types=1);

namespace Inside\Content\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContent;
use Inside\Support\Str;

final class PurgeOldContentsCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:content:purge';

    /**
     * @var string
     */
    protected $signature = 'inside:content:purge {--t|type=* : Types of content to purge}
                                                 {--k|keep=50 : number of contents to keep by content type}
                                                 {--force : force purge of contents}';

    /**
     * @var string
     */
    protected $description = 'Purge old contents. Keeps only the limit specified (default: 50)';

    private ?BridgeContent $bridgeContent = null;

    /**
     * @throws ModelSchemaNotFoundException
     */
    public function handle(): void
    {
        $contentTypes = (array) $this->option('type');

        /** @var string $keep */
        $keep = $this->option('keep');
        $keep = intval($keep);

        $force = (bool) $this->option('force');

        if (empty($contentTypes)) {
            $contentTypes = Schema::getContentTypes();
        }

        if (! $force && config('app.env') !== 'development' && ! $this->confirm('/!\ You are not in a development environment, are you sure ?')) {
            return;
        }

        collect($contentTypes)->each(function (string $contentType) use ($keep) {
            $this->purgeContentType($contentType, $keep);
            $this->getOutput()->text('------------------------------');
        });
    }

    private function getBridge(): BridgeContent
    {
        return $this->bridgeContent ??= new BridgeContent();
    }

    public function purgeContentType(string $contentType, int $keep): void
    {
        if (Str::endsWith($contentType, '_menus')) {
            $this->getOutput()->text("Skipping menu <fg=yellow>$contentType</fg=yellow>... (menu)");

            return;
        }

        if (! Schema::hasModel($contentType)) {
            $this->getOutput()->text("Invalid content type <fg=red>$contentType</fg=red> ❌");

            return;
        }

        if (in_array($contentType, config('contents.purge_protected'))) {
            $this->getOutput()->text("Skipping protected type <fg=yellow>$contentType</fg=yellow>... (in config contents.purge_protected)");

            return;
        }

        /** @var Collection $toKeep */
        $toKeep = type_to_class($contentType)::query()
            ->latest('published_at')
            ->where('langcode', config('app.locale'))
            ->take($keep)
            ->pluck('uuid_host');

        $this->getOutput()->text("Purging type <fg=yellow>$contentType</fg=yellow>...");

        /** @var Collection $toDelete */
        $toDelete = type_to_class($contentType)::whereNotIn('uuid_host', $toKeep)->pluck('uuid');

        if ($toDelete->isEmpty()) {
            $this->getOutput()->text('No content to delete');

            return;
        }

        $this->getOutput()->text("{$toDelete->count()} contents to delete");

        $this->getOutput()->progressStart($toDelete->count());

        $toDelete->each(function (string $uuid) use ($contentType) {
            $this->getBridge()->contentDelete($contentType, $uuid);
            $this->getOutput()->progressAdvance();
        });

        $this->getOutput()->progressFinish();
        $this->getOutput()->text("<fg=yellow>$contentType</fg=yellow> purged ✅");
    }
}
