<?php

declare(strict_types=1);

namespace Inside\Content\Console;

use Illuminate\Support\Arr;
use Inside\Console\Command;
use Inside\Content\Contracts\Alphavantage;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\StockPrices;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;

final class ImportStockPricesCommand extends Command
{
    protected $name = 'inside:import:stocks:prices';

    protected $signature = 'inside:import:stocks:prices {--code=*}';

    private ?BridgeContent $bridge = null;

    public function __construct(private Alphavantage $alphavantage)
    {
        parent::__construct();
    }

    public function handle(): int
    {
        if (! Schema::hasModel('stock_prices')) {
            throw ModelSchemaNotFoundException::named('stock_prices');
        }

        $codes = collect($this->option('code'));

        $stocks = StockPrices::query()
          ->when($codes->isNotEmpty(), fn (Builder $query) => $query->whereIn('stock_id', $codes))
          ->get();

        if ($stocks->isEmpty()) {
            $this->info('No stock prices to import.');

            return self::FAILURE;
        }

        $this->info("Stock prices import ({$stocks->count()})");
        $this->output->progressStart($stocks->count());
        $stocks->each(function (StockPrices $stockPrice) {
            $this->updateStockValues($stockPrice);
            $this->output->progressAdvance();
        });

        $this->output->progressFinish();
        $this->info('All stock prices have been updated');

        return self::SUCCESS;
    }

    private function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    private function updateStockValues(StockPrices $stockPrice): void
    {
        $globalQuote = $this->alphavantage->getGlobalQuote($stockPrice->stock_id);

        if (empty($globalQuote)) {
            $this->error("No global quote found for symbol $stockPrice->stock_id");

            return;
        }

        $this->getBridge()->contentUpdate($stockPrice->content_type, [
            'uuid' => $stockPrice->uuid,
            'price' => number_format(Arr::get($globalQuote, 'price'), 2),
            'change_percent' => number_format(Arr::get($globalQuote, 'change percent'), 2),
        ]);
    }
}
