<?php

namespace Inside\Content\Console;

use Illuminate\Console\Command;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use League\Csv\Reader;
use ReflectionClass;

class ImportContentTypeFromCsvCommand extends Command
{
    protected $name = 'inside:csv:content';

    protected $signature = 'inside:csv:content {csv}';

    protected $description = 'Generates content type migrations';

    /**
     * @var Filesystem
     */
    protected $files;

    public function __construct(Filesystem $files)
    {
        parent::__construct();

        $this->files = $files;
    }

    public function handle(): void
    {
        /** @var string $file */
        $file = $this->argument('csv');
        $lines = $this->getCsvRecords($file);

        $languages = list_languages();

        $labels = array_map(fn ($language) => "label - $language", $languages);
        $langLabels = array_combine($languages, $labels) ?: [];

        $descriptions = array_map(fn ($language) => "description - $language", $languages);
        $langDescriptions = array_combine($languages, $descriptions) ?: [];

        $date = date('Y_m_d', time()).'_000000';
        $contentTypes = [];
        $previous = null;

        $contentTypeName = '';
        $fieldWeight = 0;

        foreach ($lines as $key => $line) {
            switch ($line['type']) {
                case 'Type de contenu':
                    $mandatoryHeaders = [...$labels, 'name'];

                    foreach ($mandatoryHeaders as $mandatoryHeader) {
                        if (empty($line[$mandatoryHeader])) {
                            $this->error(sprintf('Missing content type %s on line %d', $mandatoryHeader, $key + 1));
                            exit(1);
                        }
                    }

                    $contentTypeName = trim($line['name']);
                    switch ($line['listing_type']) {
                        case 'Contenu':
                            $listingType = 'content';
                            break;
                        case 'Taxonomie':
                            $listingType = 'taxo';
                            break;
                        default:
                            $listingType = 'hidden';
                            break;
                    }
                    $contentTypes[$contentTypeName] = [
                        'options' => [
                            'name' => $contentTypeName,
                            'translatable' => $contentTypeName !== 'users',
                            'searchable' => (! empty($line['searchable']) && $line['searchable'] === 'TRUE') ? true : false,
                            'global_searchable' => (! empty($line['global_searchable']) && $line['global_searchable'] === 'TRUE') ? true : false,
                            'aliasable' => (! empty($line['aliasable']) && $line['aliasable'] === 'TRUE') ? true : false,
                            'permissible' => (! empty($line['permissible']) && $line['permissible'] === 'TRUE') ? true : false,
                            'categorizable' => (! empty($line['categorizable']) && $line['categorizable'] === 'TRUE') ? true : false,
                            'listing_type' => $listingType,
                            'title' => [],
                        ],
                        'fields' => [],
                    ];

                    foreach ($langLabels as $language => $label) {
                        $contentTypes[$contentTypeName]['options']['title'][$language] = trim($line[$label]);
                    }

                    $fieldWeight = 0;
                    break;
                case 'Champ':
                    $mandatoryHeaders = [...$labels, 'name', 'field_type'];

                    foreach ($mandatoryHeaders as $mandatoryHeader) {
                        if (empty($line[$mandatoryHeader])) {
                            $this->error(sprintf('Missing field %s on line %d', $mandatoryHeader, $key + 1));
                            exit(1);
                        }
                    }

                    $field = [
                        'name' => trim($line['name']),
                        'searchable' => (! empty($line['searchable']) && $line['searchable'] === 'TRUE') ? true : false,
                        'translatable' => $contentTypeName !== 'users',
                        'search_result_field' => (! empty($line['search_result_field']) && $line['search_result_field'] === 'TRUE') ? true : false,
                        'searchable_filter' => (! empty($line['searchable_filter']) && $line['searchable_filter'] === 'TRUE') ? true : false,
                        'required' => (! empty($line['required']) && $line['required'] === 'TRUE') ? true : false,
                        'weight' => $fieldWeight,
                        'title' => [],
                    ];

                    foreach ($langLabels as $language => $langLabel) {
                        $field['title'][$language] = trim($line[$langLabel]);
                    }

                    $description = [];
                    foreach ($langDescriptions as $language => $langDescription) {
                        if (isset($line[$langDescription]) && ! empty(trim($line[$langDescription]))) {
                            $description[$language] = trim($line[$langDescription]);
                        }
                    }

                    if (! empty($description)) {
                        $field['description'] = $description;
                    }

                    $fieldType = str_replace(
                        ['Téléphone', 'Texte', 'Textarea', 'Nombre', 'Paragraphes', 'Référence', 'Case à cocher', 'Image', 'Fichier', 'Date', 'Wysiwyg', 'Lien', 'Colorpicker', 'Iconpicker', 'Commentaires'],
                        ['phone', 'text', 'textarea', 'integer', 'paragraphs', 'reference', 'boolean', 'image', 'file', 'timestamp', 'wysiwyg', 'link', 'color', 'icon', 'comments'],
                        $line['field_type']
                    );
                    $method = sprintf('get%sFieldSettings', \Inside\Support\Str::studly($fieldType));

                    if (method_exists($this, $method)) {
                        $this->$method($field, $fieldWeight, $line, $key);
                    }

                    /** @var array $field */
                    $field['type'] = '{{CONSTANT_'.$this->getConstantAsString($field['type']).'}}';

                    if (array_key_exists('type', $field['widget'])) {
                        $field['widget']['type'] = '{{CONSTANT_'.$this->getConstantAsString($field['widget']['type']).'}}';
                    }

                    $contentTypes[$contentTypeName]['fields'][] = $field;

                    $fieldWeight++;
                    break;
                case 'Paragraphe':
                    $mandatoryHeaders = ['paragraph_type'];

                    foreach ($mandatoryHeaders as $mandatoryHeader) {
                        if (empty($line[$mandatoryHeader])) {
                            $this->error(sprintf('Missing paragraph %s on line %d', $mandatoryHeader, $key + 1));
                            exit(1);
                        }
                    }

                    $lastFieldKey = array_key_last($contentTypes[$contentTypeName]['fields']);
                    if ($previous['type'] === 'Champ') {
                        if ($previous['field_type'] !== 'Paragraphes') {
                            $this->error(sprintf('Paragraph type on line %d must be set for a paragraph field', $key + 1));
                            exit(1);
                        }
                        $contentTypes[$contentTypeName]['fields'][$lastFieldKey]['settings']['target_type'] = [];
                    }

                    $paragraphType = str_replace(
                        [
                            'Texte sur deux colonnes',
                            'Texte + image',
                            'Texte',
                            'Séparateur',
                            'Fichiers',
                            'Fichier',
                            'Carousel',
                            'Image',
                            'Vidéo',
                            'Carte',
                            'Accordéon',
                            'Call to action',
                            'Bouton',
                            'Code HTML',
                            'Tweet',
                            'Remontée de contenus',
                        ],
                        ['text_two_columns', 'text_with_image', 'text', 'separator', 'files', 'file', 'slider', 'image', 'video', 'map', 'accordion', 'call_to_action', 'button', 'html', 'tweet', 'fetched_contents'],
                        trim($line['paragraph_type'])
                    );

                    if (! in_array($paragraphType, Schema::getSectionTypes())) {
                        $this->error(sprintf('Paragraph type %s used on line %d does not exist', $paragraphType, $key + 1));
                        exit(1);
                    }

                    if (! in_array($paragraphType, $contentTypes[$contentTypeName]['fields'][$lastFieldKey]['settings']['target_type'])) {
                        $contentTypes[$contentTypeName]['fields'][$lastFieldKey]['settings']['target_type'][] = $paragraphType;
                    }

                    break;
                default:
                    $this->error(sprintf('Missing type on line %d, please check your csv', $key + 1));
                    exit(1);
            }

            $previous = $line;
        }

        $migrationUrl = $this->createMigrationFile();
        $bridgeMigration = '';
        foreach ($contentTypes as $name => $contentType) {
            $this->exportContents($name, $contentType, $date);
            $bridgeMigration .= str_replace('dummy_type', $name, $this->getStubBridge());
        }
        $this->writeMigrationContentType($migrationUrl, $bridgeMigration);
    }

    /**
     * @param string $constant
     * @return int|string
     */
    protected function getConstantAsString(string $constant)
    {
        $contentTypeClass = new ReflectionClass(BridgeContentType::class);
        $constants = $contentTypeClass->getConstants();

        $constName = null;
        foreach ($constants as $name => $value) {
            if ($value == $constant) {
                return $name;
            }
        }

        throw new ModelNotFoundException();
    }

    protected function getPhoneFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_PHONE_WIDGET,
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if (! empty(trim($line['default']))) {
            $field['default'] = trim($line['default']);
        }
    }

    protected function getTextFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::STRING_WIDGET,
            'settings' => [
                'size' => 60,
                'placeholder' => '',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if (! empty(trim($line['default']))) {
            $field['default'] = trim($line['default']);
        }
    }

    protected function getTextareaFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::LONGTEXT_WIDGET,
            'settings' => [
                'rows' => 5,
                'placeholder' => '',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXTAREA_FIELD;
        $field['settings'] = [
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if (! empty(trim($line['default']))) {
            $field['default'] = trim($line['default']);
        }
    }

    protected function getIntegerFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::NUMBER_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::INTEGER_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];

        if (trim($line['default']) !== '') {
            $field['default'] = (int) trim($line['default']);
        }
    }

    protected function getParagraphsFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::SECTION_WIDGET,
            'settings' => [
                'title' => 'Paragraphe',
                'title_plural' => 'Paragraphes',
                'edit_mode' => 'open',
                'add_mode' => 'dropdown',
                'form_display_mode' => 'default',
                'default_paragraph_type' => '',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::SECTION_FIELD;
        $field['settings'] = [
            'target_type' => Schema::getSectionTypes(),
            'cardinality' => -1,
        ];
    }

    protected function getReferenceFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $target = trim($line['target_type']);

        if (empty($target)) {
            $this->error('Reference field with empty target on line %d', $key + 1);
            exit(1);
        }

        if ($target !== 'users') {
            $target = [$target];
        }

        /** @var array $field */
        $field['widget'] = [
            'type' => BridgeContentType::SELECT_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];

        switch (trim($line['widget'])) {
            case 'Autocomplete':
                $field['widget']['type'] = BridgeContentType::AUTOCOMPLETE_WIDGET;
                break;
            case 'Cases à cocher':
                $field['widget']['type'] = BridgeContentType::OPTIONS_BUTTONS_WIDGET;
                break;
            case 'Caché':
                $field['widget'] = ['hidden' => true];
                break;
            case 'Liste de sélection':
            default:
                $field['widget']['type'] = BridgeContentType::SELECT_WIDGET;
                break;
        }

        $field['type'] = BridgeContentType::REFERENCE_FIELD;
        $field['settings'] = [
            'target_type' => $target,
            'cardinality' => trim($line['cardinality']) === 'Une seule' ? 1 : -1,
        ];

        if ($field['settings']['cardinality'] !== 1 && $field['widget']['type'] === BridgeContentType::SELECT_WIDGET) {
            $field['selectable_all'] = true;
        }
    }

    protected function getBooleanFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::BOOLEAN_CHECKBOX_WIDGET,
            'settings' => [
                'display_label' => true,
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::BOOLEAN_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];

        $field['default'] = false;

        if (trim($line['default']) !== '') {
            $field['default'] = (bool) trim($line['default']);
        }
    }

    protected function getImageFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::IMAGE_WIDGET,
            'settings' => [
                'progress_indicator' => 'throbber',
                'preview_image_style' => 'thumbnail',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::IMAGE_FIELD;
        $field['settings'] = [
            'uri_scheme' => 'public',
            'default_image' => [
                'uuid' => '',
                'alt' => '',
                'title' => '',
                'width' => null,
                'height' => null,
            ],
            'target_type' => 'file',
            'display_field' => false,
            'display_default' => false,
            'cardinality' => 1,
        ];
    }

    protected function getFileFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::FILE_WIDGET,
            'settings' => [
                'progress_indicator' => 'throbber',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::FILE_FIELD;
        $field['settings'] = [
            'display_field' => false,
            'display_default' => false,
            'uri_scheme' => 'public',
            'target_type' => 'file',
            'cardinality' => 1,
        ];
    }

    protected function getTimestampFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::TIMESTAMP_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TIMESTAMP_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];
    }

    protected function getWysiwygFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::WYSIWYG_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::WYSIWYG_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];
        if (! empty(trim($line['default']))) {
            $field['default'] = trim($line['default']);
        }
    }

    protected function getLinkFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_LINK_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];
    }

    protected function getColorFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_COLOR_PICKER_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if (! empty(trim($line['default']))) {
            $field['default'] = trim($line['default']);
        }
    }

    protected function getIconFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_ICON_PICKER_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];
    }

    protected function getCommentsFieldSettings(array &$field, int $fieldWeight, array $line, int $key): void
    {
        $field['widget'] = ['hidden' => true];
        $field['type'] = BridgeContentType::COMMENT_FIELD;
        $field['settings'] = [
            'comment_type' => 'comments',
            'cardinality' => 1,
        ];
    }

    /**
     * @param string $file
     * @return array
     * @throws \League\Csv\Exception
     */
    protected function getCsvRecords(string $file): array
    {
        $csv = Reader::createFromPath($file, 'r');
        $encoding = mb_detect_encoding($csv->getContent(), mb_list_encodings(), true);

        if ($encoding !== 'UTF-8') {
            $csv->setOutputBOM(Reader::BOM_UTF8);
            $csv->addStreamFilter('convert.iconv.'.$encoding.'/UTF-8');
        }

        $csv->setDelimiter(',');
        $csv->setHeaderOffset(2);

        $headers = array_map(
            function ($item) {
                $item = trim(Str::lower($item));

                return str_replace(
                    [
                        'type',
                        'intitulé',
                        'nom machine',
                        'infobulle',
                        'type de champ',
                        'type de paragraphe',
                        'type de contenu à référencer',
                        'type d\'affichage',
                        'nombre de valeurs possibles',
                        'valeur par défaut',
                        'obligatoire',
                        'aliasable',
                        'searchable',
                        'global searchable',
                        'search result field',
                        'search filter',
                        'permissible',
                        'act as category',
                        'listing type',
                    ],
                    [
                        'type',
                        'label',
                        'name',
                        'description',
                        'field_type',
                        'paragraph_type',
                        'target_type',
                        'widget',
                        'cardinality',
                        'default',
                        'required',
                        'aliasable',
                        'searchable',
                        'global_searchable',
                        'search_result_field',
                        'searchable_filter',
                        'permissible',
                        'categorizable',
                        'listing_type',
                    ],
                    $item
                );
            },
            $csv->getHeader()
        );

        $lines = iterator_to_array($csv->getRecords($headers));

        return array_slice($lines, (int) $csv->getHeaderOffset());
    }

    /**
     * @param string $name
     * @param array $contentType
     * @param string $date
     * @return void
     */
    protected function exportContents(string $name, array $contentType, string $date): void
    {
        $exportTo = cms_base_path('export/contents/'.$name);
        if (! File::exists($exportTo)) {
            File::makeDirectory($exportTo, 0755, true);
        }

        $encoder = new \Riimu\Kit\PHPEncoder\PHPEncoder();
        File::put(
            $exportTo.'/'.$date.'.php',
            '<?php

use Inside\Host\Bridge\BridgeContentType;

return '.preg_replace(
                '#\'\{\{CONSTANT_([^\}\}]+)\}\}\'#',
                'BridgeContentType::$1',
                $encoder->encode(
                    $contentType,
                    [
                        'string.escape' => false,
                    ]
                ).';'
            )
        );
    }

    /**
     * @return string
     * @throws FileNotFoundException
     */
    protected function createMigrationFile(): string
    {
        $exportTo = cms_base_path('export/migrations/');
        if (! File::exists($exportTo)) {
            File::makeDirectory($exportTo, 0755, true);
        }

        $migrationPath = $exportTo.date('Y_m_d').'_000000_create_content_type.php';
        $stub = $this->getStub();
        $stub = str_replace('DummyClass', 'CreateContentType', $stub);
        File::put($migrationPath, $stub);

        return $migrationPath;
    }

    /**
     * @param string $migrationUrl
     * @param string $bridgeMigration
     * @return void
     * @throws FileNotFoundException
     */
    protected function writeMigrationContentType(string $migrationUrl, string $bridgeMigration): void
    {
        $stub = str_replace('DummyClass', 'CreateContentType', $this->getStub());
        $stub = str_replace('{{ INSIDE_BRIDGE_MIGRATION }}', $bridgeMigration, $stub);

        File::put($migrationUrl, $stub);
    }

    /**
     * @return string
     * @throws FileNotFoundException
     */
    protected function getStub(): string
    {
        return $this->files->get(__DIR__.'/stubs/create_content_type.stub');
    }

    /**
     * @return string
     * @throws FileNotFoundException
     */
    protected function getStubBridge(): string
    {
        return $this->files->get(cms_base_path('vendor/maecia/inside/core/app/Console/Commands/stubs/bridge_create.stub'));
    }
}
