<?php

namespace Inside\Content\Console;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Inside\Console\Command;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Sections\Text;
use Inside\Host\Bridge\BridgeContent;
use League\Flysystem\FileNotFoundException;

class ImportContentFromInside1Command extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inside:import:inside1 {config} {file} {--limit=0}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Importer des contenus depuis inside1';

    /**
     * Importation config
     *
     * @var array
     */
    protected $config;

    /**
     * @var BridgeContent
     */
    protected ?BridgeContent $bridgeContent;

    /**
     * InsideImport constructor.
     */
    public function __construct(protected WysiwygImageService $wysiwygImageService)
    {
        parent::__construct();

        $this->config = config('imports');
    }

    public function handle(): void
    {
        $this->bridgeContent = new BridgeContent();
        /** @var string $configName */
        $configName = $this->argument('config');
        /** @var string $file */
        $file = $this->argument('file');
        $limit = intval($this->option('limit'));

        $supported = array_keys($this->config);

        if (! in_array($configName, $supported)) {
            $this->error(
                'The config ['.$configName.'] is not supported. You can only use supported config bellow ['.implode(
                    ', ',
                    $supported
                ).'].'
            );

            return;
        }

        if (! File::exists($file) || ! File::isFile($file) || ! File::isReadable($file)) {
            $this->error('File ['.$file.'] does not exists.');

            return;
        }

        $config = $this->config[$configName];
        $contentType = $config['content_type'];
        $fields = $config['fields'];
        $resourcesFolder = $config['resources_folder'];

        $fileContent = file_get_contents($file);
        if (! $fileContent) {
            throw new \Exception('Can\'t read json file correctly !');
        }

        $fileJsonData = json_decode($fileContent, true);
        $importedContentsCount = 0;
        foreach ($fileJsonData as $content) {
            // if the content has exactly two arrays it means that has translated version.
            if (count($content) === 2) {
                $uuid = $this->contentInsert($fields, $content[0], $contentType, $resourcesFolder);
                if (! $uuid) {
                    continue;
                }
                $this->contentInsert($fields, $content[1], $contentType, $resourcesFolder, $uuid);
            } elseif (isset($content['title'])) {
                $locale = config('app.locale');
                $content['language'] = $locale;
                $uuid = $this->contentInsert($fields, $content, $contentType, $resourcesFolder);

                $languages = collect(list_languages())->diff([$locale]);
                foreach ($languages as $language) {
                    $content['language'] = $language;
                    $this->contentInsert($fields, $content, $contentType, $resourcesFolder, $uuid);
                }
            } else {
                $this->contentInsert($fields, $content[0], $contentType, $resourcesFolder);
            }
            $importedContentsCount++;
            if ($limit && $importedContentsCount === $limit) {
                break;
            }
        }
    }

    private function contentInsert(array $fields, array $contentToImport, string $contentType, ?string $resourcesFolder = null, ?string $uuid = null): null|string
    {
        $contentData = $this->mappingFields($fields, $contentToImport, $contentType);
        if (! $contentData) {
            return null;
        }

        if ($uuid) {
            $uploadedContent = type_to_class($contentType)::find($uuid);
            $contentData['uuid_host'] = $uploadedContent->uuid_host;
            $uuid = $this->bridgeContent?->contentInsert($contentType, $contentData, true);
            $this->getOutput()->writeln('<fg=green>'.$contentData['title'].' / updated ✔</fg=green>');
        } else {
            $uuid = $this->bridgeContent?->contentInsert($contentType, $contentData, true, true);
            $this->getOutput()->writeln('<fg=green>'.$contentData['title'].' ✔</fg=green>');
        }
        $uploadedContent = type_to_class($contentType)::find($uuid);

        $this->uploadWysiwygFiles($uploadedContent, $resourcesFolder, $contentToImport['body']['wysiwygImages']);
        $this->uploadWysiwygFiles($uploadedContent, $resourcesFolder, $contentToImport['body']['wysiwygExternalImages']);

        return $uuid;
    }

    private function mappingFields(array $fields, array $content, string $contentType): null|array
    {
        $contentData = collect($fields)->mapWithKeys(function ($importKey, $insideKey) use ($content) {
            if (is_callable($importKey)) {
                $value = $importKey($content, $this->bridgeContent);
            } else {
                $value = $content[$importKey];
            }

            return [$insideKey => $value];
        })->merge([
            'type' => 'node',
            'bundle' => $contentType,
        ])->all();

        $requiredFields = collect(Schema::getFieldListing($contentType))->filter(
            fn ($fieldName) => Schema::getFieldOptions($contentType, $fieldName)['required']
                && ! in_array($fieldName, ['authors', 'comments', 'status', 'content'])
        );

        $requiredFieldsNotFound = collect($requiredFields)->filter(fn ($fieldName) => ! isset($contentData[$fieldName]));
        if ($requiredFieldsNotFound->isNotEmpty()) {
            $requiredFieldsNotFound->each(fn ($fieldName) => $this->getOutput()->writeln('<fg=red> Required field "'.$fieldName.'" does not exists x</fg=red>'));

            return null;
        }

        return $contentData;
    }

    private function uploadWysiwygFiles(Content $importedContent, ?string $resourcesFolder, ?array $wysiwygImages): void
    {
        if (! $wysiwygImages) {
            return;
        }

        $storage = Storage::disk('local');
        $urls = [];
        foreach ($wysiwygImages as $wysiwygImage) {
            try {
                $fileName = $wysiwygImage['fileName'];
                $path = $resourcesFolder.'/'.$fileName;
                $file = new UploadedFile(
                    $storage->path($path),
                    $fileName,
                    $storage->mimeType($path)
                );
                $upload = $this->wysiwygImageService->upload($file);
                $urls[] = $upload;

                /** @var Collection $contentSections */
                $contentSections = $importedContent->section_content;
                foreach ($contentSections as $contentSection) {
                    if (get_class($contentSection) !== Text::class) {
                        continue;
                    }

                    $srcParts = explode('/', $upload);
                    $uploadedFileName = end($srcParts);
                    $htmlBody = str_replace(
                        '/'.$fileName,
                        '/'.$uploadedFileName,
                        $contentSection->body
                    );

                    $contentSection->update([
                        'body' => $htmlBody,
                    ]);
                }
            } catch (FileNotFoundException $exception) {
                $this->getOutput()->writeln('<fg=red> File "'.$fileName.'" not found ! x</fg=red>');
                continue;
            }
        }

        if (! empty($urls)) {
            $this->wysiwygImageService->store($importedContent, $urls);
        }
    }
}
