<?php

namespace Inside\Content\Console;

use Carbon\Carbon;
use Inside\Console\Command;
use Inside\Content\Facades\Schema;
use Inside\Content\Jobs\GenerateStylizedImage;
use Inside\Content\Jobs\UpdateImage;
use Inside\Content\Models\Contents\ImageStyles;

/**
 * Class GenerateAllImageCommand
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class GenerateAllImageCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:images:generate';

    /**
     * @var string
     */
    protected $signature = 'inside:images:generate';

    /**
     * @var string
     */
    protected $description = 'Generate all images with Images Styles';

    /**
     * Handle index command
     *
     * @return void
     * @throws \Exception
     */
    public function handle(): void
    {
        $now = Carbon::now();

        $this->info('Starting to generate at ['.$now->toFormattedDateString().' '.$now->toTimeString().']');

        $imageFields = Schema::getAllFieldsListingOfType('image');

        // Prepare needes types
        $styles = [];
        foreach ($imageFields as $modelName => $fields) {
            foreach ($fields as $fieldName) {
                $options = Schema::getFieldOptions($modelName, $fieldName);
                if (! isset($options['image_styles'])) {
                    continue;
                }
                $styles[$modelName][$fieldName] = ImageStyles::query()->whereIn('title', $options['image_styles'])->get();
            }
        }
        // Rebuild images
        $count = 0;
        foreach ($imageFields as $modelName => $fields) {
            $query = call_user_func(
                (Schema::isSectionType($modelName) ? section_type_to_class($modelName) : type_to_class($modelName))
                .'::query'
            );
            $contents = $query->where('status', true)->get();
            foreach ($contents as $content) {
                foreach ($fields as $fieldName) {
                    $image = $content->{$fieldName};
                    if ($image && isset($styles[$modelName])
                        && isset($styles[$modelName][$fieldName])
                    ) {
                        foreach ($styles[$modelName][$fieldName] as $style) {
                            GenerateStylizedImage::dispatch($image, $style)->onQueue(get_images_queue_name());
                            $count++;
                        }
                    }
                }
            }
        }
        $this->line('');
        $this->info(
            'Generation done ( '.$count.' image(s) in generation engine), started ['.$now->diffForHumans().']'
        );
    }
}
