<?php

namespace Inside\Content\Console;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\File;
use Inside\Console\Command;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use ReflectionClass;

/**
 * Class ExportSectionTypeCommand
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ExportSectionTypeCommand extends Command
{
    protected $name = 'inside:section:export';

    protected $signature = 'inside:section:export {section}';

    protected $description = 'Export un type de paragraphe';

    public function handle(): void
    {
        /** @var string $section */
        $section = $this->argument('section');
        if ($section !== null && Schema::hasSectionType($section)) {
            $options = Schema::getModelOptions($section);
            if ($options['name'] != $section) {
                $this->error('Mauvais nom');
                exit(-1);
            }

            // Get groups infos
            $sectionDescription = [
                'options' => [
                    'translatable'      => (bool) ($options['translatable'] ?? false),
                    'searchable'        => (bool) ($options['searchable'] ?? false),
                    'global_searchable' => (bool) ($options['global_searchable'] ?? false),
                    'aliasable'         => (bool) ($options['aliasable'] ?? false),
                    'permissible'       => (bool) ($options['permissible'] ?? false),
                    'categorizable'     => (bool) ($options['categorizable'] ?? false),
                    'name'              => $options['name'],
                    'title'             => [],
                ],
                'fields'  => [],
            ];

            foreach ($options['title'] as $language => $text) {
                $sectionDescription['options']['title'][$language] = $text;
            }
            $bridge = new BridgeContentType();

            foreach (Schema::getFieldListing($section) as $field) {
                $options = Schema::getFieldOptions($section, $field);

                // Reload storage info to be sure !
                try {
                    $settings = $bridge->getDrupalStorageSettings($section, $field, 'paragraph');
                } catch (\Exception $e) {
                    continue; // We didn't get setting ( this often means that we lost synchronisation between inside / drupal )
                }

                if ($settings !== false) {
                    $fieldDescription = [
                        'name'                => $field,
                        'searchable'          => (bool) ($options['searchable'] ?? false),
                        'translatable'        => (bool) ($options['translatable'] ?? false),
                        'default'             => (bool) ($options['default'] ?? ''),
                        'search_result_field' => (bool) ($options['search_result_field'] ?? false),
                        'required'            => (bool) ($options['required'] ?? false),
                        'weight'              => $options['weight'],
                        'title'               => [],
                        'description'         => [],
                        'widget'              => $bridge->getFormWidgetOptions($section, $field),
                    ];

                    foreach ($options['title'] as $language => $text) {
                        $fieldDescription['title'][$language] = $text;
                    }

                    foreach ($options['description'] as $language => $text) {
                        $fieldDescription['description'][$language] = $text;
                    }

                    $fieldDescription = array_merge($fieldDescription, $settings);

                    $fieldDescription['type'] =
                        '{{CONSTANT_'.$this->getConstantAsString($fieldDescription['type']).'}}';

                    if ($fieldDescription['widget']['type']) {
                        $fieldDescription['widget']['type'] =
                            '{{CONSTANT_'.$this->getConstantAsString($fieldDescription['widget']['type']).'}}';
                    }
                    $sectionDescription['fields'][] = $fieldDescription;
                }
            }

            $exportTo = cms_base_path('export/sections/'.$section);
            if (! File::exists($exportTo)) {
                File::makeDirectory($exportTo, 0755, true);
            }
            $encoder = new \Riimu\Kit\PHPEncoder\PHPEncoder();
            File::put(
                $exportTo.'/'.Carbon::now()->format('Y_m_d_his').'.php',
                '<?php
use Inside\Host\Bridge\BridgeContentType;
return '.preg_replace(
                    '#\'\{\{CONSTANT_([^\}\}]+)\}\}\'#',
                    'BridgeContentType::$1',
                    $encoder->encode(
                        $sectionDescription,
                        [
                            'string.escape' => false,
                        ]
                    ).';'
                )
            );
            $this->info('Le type de section ['.$section.'] a correctement exporté dans ['.$exportTo.']');
        } else {
            $this->error('La section ['.$section.'] n\'existe pas');
            $this->table(
                ['Accepted types'],
                array_map(
                    function ($value) {
                        return [$value];
                    },
                    Schema::getSectionTypes()
                )
            );
        }
    }

    /**
     * @param string $constant
     * @return int|string
     */
    protected function getConstantAsString(string $constant)
    {
        $contentTypeClass = new ReflectionClass(BridgeContentType::class);
        $constants = $contentTypeClass->getConstants();

        $constName = null;
        foreach ($constants as $name => $value) {
            if ($value == $constant) {
                return $name;
            }
        }

        throw new ModelNotFoundException();
    }
}
