<?php

namespace Inside\Content\Console;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\File;
use Inside\Console\Command;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use ReflectionClass;

/**
 * Class ExportContentTypeCommand
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ExportContentTypeCommand extends Command
{
    protected $name = 'inside:type:export';

    protected $signature = 'inside:type:export {type}';

    protected $description = 'Export un type de contenu';

    public function handle(): void
    {
        /** @var string $type */
        $type = $this->argument('type');
        if ($type !== null && Schema::hasContentType($type)) {
            $options = Schema::getModelOptions($type);
            if ($options['name'] != $type) {
                $this->error('Erreur de nommage');
                exit(-1);
            }

            // Get groups infos
            $typeDescription = [
                'options' => [
                    'translatable'      => (bool) ($options['translatable'] ?? false),
                    'searchable'        => (bool) ($options['searchable'] ?? false),
                    'global_searchable' => (bool) ($options['global_searchable'] ?? false),
                    'aliasable'         => (bool) ($options['aliasable'] ?? false),
                    'permissible'       => (bool) ($options['permissible'] ?? false),
                    'categorizable'     => (bool) ($options['categorizable'] ?? false),
                    'name'              => $options['name'],
                    'title'             => [],
                ],
                'fields'  => [],
            ];

            foreach ($options['title'] as $language => $text) {
                $typeDescription['options']['title'][$language] = $text;
            }
            $bridge = new BridgeContentType();

            // Get groups informations
            $groups = [
                BridgeContentType::GROUP_PUBLISHING_OPTIONS['name'] => $bridge->getFormGroupOptions(
                    $type,
                    BridgeContentType::GROUP_PUBLISHING_OPTIONS['name']
                ),
                BridgeContentType::GROUP_ADVANCED_OPTIONS['name']   => $bridge->getFormGroupOptions(
                    $type,
                    BridgeContentType::GROUP_ADVANCED_OPTIONS['name']
                ),
            ];

            foreach (Schema::getFieldListing($type) as $field) {
                $options = Schema::getFieldOptions($type, $field);

                // Reload storage info to be sure !
                try {
                    $settings = $bridge->getDrupalStorageSettings($type, $field);
                } catch (\Exception $e) {
                    continue; // We didn't get setting ( this often means that we lost synchronisation between inside / drupal )
                }

                if ($settings !== false) {
                    $fieldDescription = [
                        'name'                => $field,
                        'searchable'          => (bool) ($options['searchable'] ?? false),
                        'translatable'        => (bool) ($options['translatable'] ?? false),
                        'default'             => (bool) ($options['default'] ?? ''),
                        'search_result_field' => (bool) ($options['search_result_field'] ?? false),
                        'required'            => (bool) ($options['required'] ?? false),
                        'weight'              => $options['weight'],
                        'title'               => [],
                        'description'         => [],
                        'widget'              => $bridge->getFormWidgetOptions($type, $field),
                    ];

                    foreach ($options['title'] as $language => $text) {
                        $fieldDescription['title'][$language] = $text;
                    }

                    foreach ($options['description'] as $language => $text) {
                        $fieldDescription['description'][$language] = $text;
                    }

                    $fieldDescription = array_merge($fieldDescription, $settings);

                    if ($groups[BridgeContentType::GROUP_PUBLISHING_OPTIONS['name']]
                        && in_array(
                            $field,
                            $groups[BridgeContentType::GROUP_PUBLISHING_OPTIONS['name']]['children']
                        )
                    ) {
                        $fieldDescription['widget']['group'] = BridgeContentType::GROUP_PUBLISHING_OPTIONS['name'];
                    } elseif ($groups[BridgeContentType::GROUP_ADVANCED_OPTIONS['name']]
                        && in_array(
                            $field,
                            $groups[BridgeContentType::GROUP_ADVANCED_OPTIONS['name']]['children']
                        )
                    ) {
                        $fieldDescription['widget']['group'] = BridgeContentType::GROUP_ADVANCED_OPTIONS['name'];
                    }

                    $fieldDescription['type'] =
                        '{{CONSTANT_'.$this->getConstantAsString($fieldDescription['type']).'}}';

                    if ($fieldDescription['widget']['type']) {
                        $fieldDescription['widget']['type'] =
                            '{{CONSTANT_'.$this->getConstantAsString($fieldDescription['widget']['type']).'}}';
                    }
                    $typeDescription['fields'][] = $fieldDescription;
                }
            }

            $exportTo = cms_base_path('export/contents/'.$type);
            if (! File::exists($exportTo)) {
                File::makeDirectory($exportTo, 0755, true);
            }
            $encoder = new \Riimu\Kit\PHPEncoder\PHPEncoder();
            File::put(
                $exportTo.'/'.Carbon::now()->format('Y_m_d_his').'.php',
                '<?php
use Inside\Host\Bridge\BridgeContentType;
return '.preg_replace(
                    '#\'\{\{CONSTANT_([^\}\}]+)\}\}\'#',
                    'BridgeContentType::$1',
                    $encoder->encode(
                        $typeDescription,
                        [
                            'string.escape' => false,
                        ]
                    ).';'
                )
            );
            $this->info('Le type de contenu ['.$type.'] a correctement exporté dans ['.$exportTo.']');
        } else {
            $this->error('Type ['.$type.'] dn\'existe pas ');
            $this->table(
                ['Accepted types'],
                array_map(
                    function ($value) {
                        return [$value];
                    },
                    Schema::getContentTypes()
                )
            );
        }
    }

    /**
     * @param string $constant
     * @return int|string
     */
    protected function getConstantAsString(string $constant)
    {
        $contentTypeClass = new ReflectionClass(BridgeContentType::class);
        $constants = $contentTypeClass->getConstants();

        $constName = null;
        foreach ($constants as $name => $value) {
            if ($value == $constant) {
                return $name;
            }
        }

        throw new ModelNotFoundException();
    }
}
