<?php

namespace Inside\Content\Console;

use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;
use Inside\Support\Str;

class DeleteContentCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:content:delete';

    /**
     * @var string
     */
    protected $signature = 'inside:content:delete {type : The type of content}
                                                  {--F|fieldName=* : example with a normal where (-F title) example with a reference for the where (-F title-NewsCategories)}
                                                  {--N|name=*? : The value to make the condition, for a string of characters separate the characters by _ (-N Mon_titre_long)}';

    /**
     * @var string
     */
    protected $description = 'Delete of all the contents of a content type';

    /**
     * @throws ModelSchemaNotFoundException
     */
    public function handle(): void
    {
        /** @var string $contentType */
        $contentType = $this->argument('type');
        /** @var array $fieldName */
        $fieldName = $this->option('fieldName');
        /** @var array $name */
        $name = $this->option('name') ?? [];

        if (! Schema::hasContentType($contentType)) {
            throw new ModelSchemaNotFoundException("There is no model named `{$contentType}`.");
        }

        $dataCondition = [];

        for ($i = 0; $i < count($fieldName); $i++) {
            $dataCondition[$fieldName[$i]] = $name[$i];
        }

        $contents = $this->getContents($dataCondition, $contentType);

        $bridge = new BridgeContent();
        $bar = $this->getOutput()->createProgressBar($contents->count());
        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%");
        $contents = $contents->get();
        /** @var \Illuminate\Support\Collection<Content> $contents */
        foreach ($contents as $content) {
            if (is_null($content->uuid)) {
                continue;
            }
            $bridge->contentDelete($contentType, $content->uuid);
            $bar->setMessage(sprintf('<comment>Deleting %s content [%s] (%s)</comment>', $contentType, trim($content->title), $content->uuid));
            $bar->advance();
        }
    }

    /**
     * @param array $dataCondition
     * @param string $contentType
     *
     * @return Builder
     */
    protected function getContents(array $dataCondition, string $contentType): Builder
    {
        $contents = call_user_func(type_to_class($contentType).'::query');
        if (! empty($dataCondition)) {
            $contents = $this->getContentWithConditions($dataCondition, $contents, $contentType);
        }

        return $contents;
    }

    /**
     * @param array $dataCondition
     * @param Builder $contents
     * @param string $contentType
     * @return Builder
     */
    protected function getContentWithConditions(array $dataCondition, Builder $contents, string $contentType): Builder
    {
        foreach ($dataCondition as $field => $value) {
            $value = Str::replace('_', ' ', $value);
            if (Str::contains($field, '-')) {
                $fieldReference = explode('-', $field);
                if (! Schema::hasField($contentType, $fieldReference[0])) {
                    throw new FieldSchemaNotFoundException("There is no field named `{$fieldReference[0]}` in model `{$contentType}`.");
                }

                $contents->whereHas($fieldReference[1], function ($contents) use ($value, $fieldReference) {
                    $contents->where('inside_content_'.class_to_type($fieldReference[1]).'.'.$fieldReference[0], $value);
                });
            } else {
                if (! Schema::hasField($contentType, $field)) {
                    throw new FieldSchemaNotFoundException("There is no field named `{$field}` in model `{$contentType}`.");
                }
                $contents->where($field, $value);
            }
        }

        return $contents;
    }
}
