<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Contents\Custom;
use Inside\Content\Models\Contents\ToolsCategories;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;
use Inside\Menu\Models\HeaderMenuLink;
use Inside\Slug\Models\Slug;

class ImproveToolsListing extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);

        if (in_array(strtolower(config('app.code')), ['gsia', 'imer', 'syct'])) {
            $this->console->writeln('Skipping migration for GSIA, IMER and SYCT projects');
            Log::debug('[ImproveToolsListingMigration] Skipping migration for GSIA, IMER, EPAR and SYCT projects');

            return;
        }

        $this->console->writeln('Checking if tools_categories_listings content type exists');
        Log::debug('[ImproveToolsListingMigration] Checking if tools_categories_listings content type exists');
        if (InsideSchema::hasContentType('tools_categories_listings')) {
            $this->console->writeln('Checking if tools_categories_listings has data');
            Log::debug('[ImproveToolsListingMigration] Checking if tools_categories_listings has data');
            if (DB::table('inside_content_tools_categories_listings')->exists()) {
                return;
            }
        }

        $this->console->writeln('Checking if tools custom page exists');
        Log::debug('[ImproveToolsListingMigration] Checking if tools custom page exists');
        $customPage = Custom::where('redirection_page', 'ToolsPage')->get();
        if ($customPage->isEmpty()) {
            return;
        }

        $this->console->writeln('Checking if tools content type exists');
        Log::debug('[ImproveToolsListingMigration] Checking if tools content type exists');
        if (! InsideSchema::hasContentType('tools')) {
            return;
        }

        $this->console->writeln('Checking if tools_categories content type exists');
        Log::debug('[ImproveToolsListingMigration] Checking if tools_categories content type exists');
        if (! InsideSchema::hasContentType('tools_categories')) {
            return;
        }

        $bridge = new BridgeContentType();
        $bridgeContent = new BridgeContent();

        $this->console->writeln('Creating tools_categories_listings content type');
        Log::debug('[ImproveToolsListingMigration] Creating tools_categories_listings content type');
        $this->migrateContent('tools_categories_listings');

        $this->console->writeln('Creating tools_categories_listings field');
        Log::debug('[ImproveToolsListingMigration] Creating tools_categories_listings field');
        $bridge->contentTypeCreateOrUpdateField(
            type: 'tools_categories',
            fieldName: 'tools_categories_listings',
            options:[
                'name' => 'tools_categories_listings',
                'searchable' => true,
                'translatable' => true,
                'search_result_field' => false,
                'searchable_filter' => false,
                'required' => true,
                'weight' => -2,
                'title' => [
                    'fr' => 'Listing catégorie d\'outils',
                    'en' => 'Tools categories listing',
                    'es' => 'Listado de categorías de herramientas',
                    'pt' => 'Listagem de categorias de ferramentas',
                    'de' => 'Werkzeugkategorienliste',
                    'nl' => 'Lijst van gereedschapscategorieën',
                ],
                'widget' => [
                    'type' => BridgeContentType::SELECT_WIDGET,
                    'settings' => [],
                    'weight' => 1,
                ],
                'type' => BridgeContentType::REFERENCE_FIELD,
                'settings' => [
                    'target_type' => ['tools_categories_listings'],
                    'cardinality' => 1,
                ],
            ],
            weight: -2
        );

        $toolsLang = [
            'fr' => 'Outils',
            'en' => 'Tools',
            'es' => 'Herramientas',
            'pt' => 'Ferramentas',
            'de' => 'Werkzeuge',
            'nl' => 'Gereedschap',
        ];

        $langs = array_unique(array_merge([config('app.locale', 'fr')], list_languages()));

        foreach ($langs as $lang) {
            $this->console->writeln('Updating custom page slug to '.$toolsLang[$lang].'_old');
            Log::debug('[ImproveToolsListingMigration] Updating custom page slug to '.$toolsLang[$lang].'_old');
            Slug::where('slug', strtolower($toolsLang[$lang]))
                ->update(['slug' => strtolower($toolsLang[$lang]).'_old']);

            $this->console->writeln('Updating tools_categories_listings uuid_host');
            Log::debug('[ImproveToolsListingMigration] Updating tools_categories_listings uuid_host');
            $uuidHost = DB::table('inside_content_tools_categories_listings')
                ->where('langcode', config('app.locale', 'fr'))
                ->first()
                ->uuid_host;

            $this->console->writeln('Creating tools_categories_listings page for '.$lang);
            Log::debug('[ImproveToolsListingMigration] Creating tools_categories_listings page for '.$lang);
            $data = [
                'title' => $toolsLang[$lang],
                'langcode' => $lang,
            ];

            if ($uuidHost) {
                $data['uuid_host'] = $uuidHost;
            }

            $toolsCategoriesListings = $bridgeContent->contentInsert(
                type: 'tools_categories_listings',
                data: $data
            );

            $this->console->writeln('Add all tools categories to the default tools_categories_listings page for lang: '.$lang);
            Log::debug('[ImproveToolsListingMigration] Add all tools categories to the default tools_categories_listings page for lang: '.$lang);
            $toolsCategories = ToolsCategories::where('langcode', $lang)
                ->pluck('uuid')
                ->each(function ($uuid) use ($toolsCategoriesListings, $bridgeContent) {
                    $bridgeContent->contentInsert(
                        type: 'tools_categories',
                        data: [
                            'uuid' => $uuid,
                            'tools_categories_listings' => $toolsCategoriesListings,
                        ]
                    );
                });

            $this->console->writeln('Updating header menu link for lang: '.$lang);
            Log::debug('[ImproveToolsListingMigration] Updating header menu link for lang: '.$lang);
            $menuHeader = HeaderMenuLink::query()->where('page', 'ToolsPage')
                ->where('langcode', $lang)
                ->get();

            if ($menuHeader->isNotEmpty()) {
                $menuHeader->each(fn ($menu) => $menu->update([
                    'link' => 'tools_categories_listings|'.$toolsCategoriesListings,
                ]));
            }
        }

        $this->console->writeln('Disabling old custom page');
        Log::debug('[ImproveToolsListingMigration] Disabling old custom page');
        if ($customPage->isNotEmpty()) {
            $customPage->each(fn ($page) => $page->update(['status' => 0]));
        }

        $this->console->writeln('Rebuilding models');
        Log::debug('[ImproveToolsListingMigration] Rebuilding models');
        Artisan::call('models:rebuild');
        $this->finishUp();
    }
}
