<?php

namespace Inside\Authentication\Models;

use Carbon\CarbonInterval;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Query\Builder;
use Inside\Authentication\Facades\InsideSessionLifetime;

/**
 * Inside\Authentication\Models\Token
 *
 * @property int $id
 * @property string $authenticator
 * @property string $user_uuid
 * @property string $token
 * @property \Illuminate\Support\Carbon|null $last_used_at
 * @property array|null $abilities
 * @property array|null $device
 * @property string|null $last_login_ip
 * @property string $plainTextToken
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property bool $longer_lifetime
 * @property-read mixed $created_at_timestamp
 * @property-read mixed $has_expired
 * @property-read mixed $last_used_at_timestamp
 * @property-read User $user
 * @method static \Illuminate\Database\Eloquent\Builder|Token active()
 * @method static \Illuminate\Database\Eloquent\Builder|Token newModelQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|Token newQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|Token query()
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereAbilities($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereAuthenticator($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereDevice($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereLastLoginIp($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereLastUsedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereLongerLifetime($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereToken($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereUpdatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token whereUserUuid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|Token select(array|mixed $columns)
 * @mixin \Eloquent
 * @mixin Builder
 */
class Token extends Model
{
    protected $table = 'inside_authentication_tokens';

    protected $casts = [
        'abilities' => 'json',
        'device' => 'json',
        'last_used_at' => 'datetime',
        'expires_at' => 'datetime',
        'longer_lifetime' => 'boolean',
    ];

    protected $fillable = [
        'name',
        'token',
        'device',
        'abilities',
        'authenticator',
        'last_login_ip',
        'longer_lifetime',
    ];

    protected $hidden = [
        'token',
    ];

    protected $appends = [
        'created_at_timestamp',
        'last_used_at_timestamp',
        'has_expired',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_uuid', 'uuid');
    }

    public function cant(string $ability): bool
    {
        return ! $this->can($ability);
    }

    public function can(string $ability): bool
    {
        return in_array('*', $this->abilities ?? [])
            || array_key_exists($ability, array_flip($this->abilities ?? []));
    }

    public function getHasExpiredAttribute(): bool
    {
        return $this->hasExpired();
    }

    public function getAuthenticatorLifeTime(): ?int
    {
        return InsideSessionLifetime::getAuthenticatorLifetime($this->authenticator, $this->longer_lifetime);
    }

    public function hasExpired(): bool
    {
        $interval = $this->getAuthenticatorLifeTime();

        if ($interval === null) {
            return false;
        }

        return (bool) $this->last_used_at?->lte(now()->subMinutes($interval));
    }

    public function isAboutToFade(): bool
    {
        $interval = $this->getAuthenticatorLifeTime();

        if ($interval === null) {
            return false;
        }

        return (bool) $this->last_used_at?->lte(now()->subMinutes($interval + 2));
    }

    public function getCreatedAtTimestampAttribute(): ?int
    {
        // @phpstan-ignore-next-line
        return $this->created_at?->timestamp;
    }

    public function getLastUsedAtTimestampAttribute(): ?int
    {
        // @phpstan-ignore-next-line
        return $this->last_used_at?->timestamp;
    }

    public function scopeActive(Builder $query): Builder
    {
        $longLifeTime = InsideSessionLifetime::getSessionLongerLifetime();
        $shortLifeTime = InsideSessionLifetime::getSessionLifetime();

        return $query->whereNotNull('last_used_at')
            ->where(function ($query) use ($shortLifeTime, $longLifeTime) {
                $query->where(function ($query) use ($shortLifeTime) {
                    $query->where('last_used_at', '>', now()->subMinutes($shortLifeTime))
                        ->where('longer_lifetime', false);
                })->orWhere(function ($query) use ($longLifeTime) {
                    $query->where('last_used_at', '>', now()->subMinutes($longLifeTime))
                        ->where('longer_lifetime', true);
                });
            });
    }
}
