<?php

namespace Inside\Authentication\Http\Controllers\Otp;

use DateTime;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Authentication\Contracts\Authentication as AuthenticationContract;
use Inside\Authentication\Http\Controllers\Controller;
use Inside\Authentication\Models\OtpToken;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;

class Check extends Controller
{
    public function __invoke(Request $request, AuthenticationContract $authentication): JsonResponse
    {
        $credentials = $request->only('uuid', 'otp_token');
        if (! isset($credentials['uuid']) || ! isset($credentials['otp_token'])) {
            return response()->json(['message' => 'Missing required parameters !'], 400);
        }

        $otpToken = OtpToken::where('user_uuid', $credentials['uuid'])
            ->where('token', (string) $credentials['otp_token'])
            ->get();

        if (! count($otpToken)) {
            return response()->json(['message' => 'Invalid token !'], 401);
        }

        $dateNow = new DateTime();
        $dateNow->setTimezone(new \DateTimeZone('UTC'));
        if (
            count($otpToken) !== 1
            || $otpToken[0]->created_at->diffInMinutes($dateNow) > (int) env('2FA_TOKEN_EXPIRATION_TIME', 10)
        ) {
            OtpToken::whereUserUuid($credentials['uuid'])->delete();

            return response()->json(['message' => 'Expired token !'], 401);
        }

        $token = $authentication->logAs($otpToken[0]->user, 'ldap');
        $user = $token->accessToken->user;

        $transformed = $user->toArray();
        $transformed['api_token'] = $token->plainTextToken;
        $transformed['permissions'] = [
            'roles' => Role::listUserRoles($user),
            'add' => Permission::getAllowedCreationTypesForUser($user),
            'custom' => Permission::customList($user),
            'backoffice_entries' => Permission::backofficeAccessibleEntries($user),
        ];

        // Set header so we can use auth api guard
        request()->headers->set('api_token', $token->plainTextToken);

        // Delete used otp token
        $otpToken[0]->delete();

        return response()->json(['data' => $transformed]);
    }
}
