<?php

declare(strict_types=1);

namespace Inside\Authentication\Http\Controllers\Auth;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Contracts\Authentication as AuthenticationContract;
use Inside\Authentication\Http\Controllers\Controller;
use Inside\Authentication\Models\User;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;
use InvalidArgumentException;

final class LoginAs extends Controller
{
    /**
     * @throws AuthorizationException
     * @throws InvalidArgumentException
     */
    public function __invoke(Request $request, AuthenticationContract $authentication): JsonResponse
    {
        if (! is_maecia_admin()) {
            throw new AuthorizationException(Lang::get('forbidden.notAuthorized'));
        }
        $requestUuid = $request->get('uuid');
        if ($requestUuid === null) {
            throw new InvalidArgumentException('invalid argument');
        }

        // Load user
        $user = User::findOrFail($requestUuid);

        // Check user is not a super admin
        if ($user->hasAllRoles('super_administrator') && $request->user()->email !== config('app.technical_mail')) {
            throw new AuthorizationException(Lang::get('forbidden.logas.superadmin'));
        }

        $token = $authentication->logAs($user, 'inside-admin');
        $user = $token->accessToken->user;

        $transformed = $user->toArray();
        $transformed['api_token'] = $token->plainTextToken;
        $transformed['permissions'] = [
            'roles' => Role::listUserRoles($user),
            'add' => Permission::getAllowedCreationTypesForUser($user),
            'custom' => Permission::customList($user),
            'backoffice_entries' => Permission::backofficeAccessibleEntries($user),
        ];

        // Set header so we can use auth api guard
        $request->headers->set('api_token', $token->plainTextToken);

        return response()->json(['data' => $transformed]);
    }
}
