<?php

namespace Inside\Ai\Services\Providers;

use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Ai\Contracts\AIProviderContract;
use Inside\Ai\DTO\AIResponse;
use Inside\Ai\Exceptions\AIConfigurationException;
use Inside\Ai\Exceptions\AIProviderException;
use Inside\Facades\Http;

class AzureOpenAIProvider implements AIProviderContract
{
    private const DEFAULT_MAX_TOKEN = 4096;
    private const DEFAULT_TEMPERATURE = 0.3;
    private const TEMPERATURES = [
        'correct' => 0.2,
        'rewrite' => 0.3,
        'formalize' => 0.4,
        'shortening' => 0.5,
        'analize' => 0.6
    ];

    public function __construct(
        protected readonly string $apiKey,
        protected readonly string $apiUrl,
        protected readonly string $deploymentName,
    ) {
    }

    /**
     * Process text with AI
     */
//    public function processText(string $text, string $prompt, array $options = []): AIResponse
    public function processText(string $text, string $prompt, array $options = []): AIResponse
    {
        try {
            $model = $this->deploymentName;
            $maxTokens = $options['max_output_tokens'] ?? self::DEFAULT_MAX_TOKEN;
            $temperature = $this->getTemperature($options);

            $messages = [
                ['role' => 'system', 'content' => $prompt],
                ['role' => 'user',   'content' => $text]
            ];

            $endpoint = rtrim($this->apiUrl, '/') . "/openai/deployments/{$this->deploymentName}/chat/completions?api-version=2023-05-15";

            $response = Http::withHeaders([
                'api-key' => $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post($endpoint, [
                'messages' => $messages,
                'temperature' => $temperature,
                'max_tokens' => $maxTokens,
            ]);

            if (!$response->successful()) {
                Log::error('Azure OpenAI API Error', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                throw new AIProviderException(Lang::get('validation.error_call_api', ['body' => $response->body()]));
            }

            $data = $response->json();

            $processedText = null;
            if (!empty(data_get($data, 'output_text'))) {
                $processedText = trim(data_get($data, 'output_text'));
            }

            if ($processedText === null && is_array(data_get($data, 'output'))) {
                $parts = [];
                foreach (data_get($data, 'output', []) as $out) {
                    if (isset($out['content']) && is_array($out['content'])) {
                        foreach ($out['content'] as $c) {
                            // différents formats possibles
                            if (isset($c['type']) && ($c['type'] === 'output_text' || $c['type'] === 'text')) {
                                $parts[] = trim($c['text'] ?? $c['value'] ?? '');
                            } elseif (isset($c['type']) && $c['type'] === 'message' && isset($c['text'])) {
                                $parts[] = trim($c['text']);
                            } elseif (isset($c['text'])) {
                                $parts[] = trim($c['text']);
                            } elseif (is_string($c)) {
                                $parts[] = trim($c);
                            }
                        }
                    }
                    if (isset($out['text']) && is_string($out['text'])) {
                        $parts[] = trim($out['text']);
                    }
                    if (isset($out['message']['content']) && is_string($out['message']['content'])) {
                        $parts[] = trim($out['message']['content']);
                    }
                }
                $processedText = trim(implode("\n\n", array_filter($parts)));
            }

            if ($processedText === null && !is_null(data_get($data, 'choices.0.message.content'))) {
                $processedText = trim(data_get($data, 'choices.0.message.content'));
            }

            if (empty($processedText)) {
                Log::error('Azure OpenAI: cannot parse response', ['response' => $data]);
                throw new AIProviderException(Lang::get('validation.invalide_openai_response'));
            }

            $promptTokens = data_get($data, 'usage.prompt_tokens', data_get($data, 'usage.promptTokenCount', 0));
            $completionTokens = data_get($data, 'usage.completion_tokens', data_get($data, 'usage.completionTokenCount', 0));

            return AIResponse::success(
                processedText: $processedText,
                metadata: [
                    'model' => $model,
                    'prompt_tokens' => $promptTokens,
                    'completion_tokens' => $completionTokens,
                ]
            );

        } catch (AIProviderException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::error('Azure OpenAI Processing Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return AIResponse::failure(Lang::get('validation.openai_error_process', ['message' => $e->getMessage()]));
        }
    }

    private function getTemperature($option): float
    {
        return data_get(self::TEMPERATURES, $option['action'], self::DEFAULT_TEMPERATURE);
    }
}
