<?php

declare(strict_types=1);

namespace Inside\Workflow\Services;

use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\ContentCache;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Facades\Package;
use Inside\Host\Bridge\BridgeContent;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;
use Inside\Workflow\Events\ProposalEditedEvent;
use Inside\Workflow\Events\ProposalReviewedEvent;
use Inside\Workflow\Models\Proposal;
use Inside\Workflow\Models\ProposalStep;
use Inside\Workflow\Models\Workflow;
use Inside\Workflow\Models\WorkflowStep;

final class ProposalService
{
    public function getContentAuthorUuid(Content $content): string
    {
        return $content->update_author ?? $content->author; // @phpstan-ignore-line
    }

    public function bypassWorkflow(?Content $content): bool
    {
        if (! $content || in_array($content->content_type, ['users', 'main_menus', 'shortcuts_menus', 'footer_menus', 'comments'])) {
            return true;
        }

        /** @var User $author */
        $author = User::find($this->getContentAuthorUuid($content));

        if ($author->roles()->where(['name' => 'super_administrator'])->count()) {
            return true;
        }

        $customBypass = config('workflow.bypass_workflow', false);

        if ($customBypass && is_callable($customBypass) && $customBypass($content)) {
            return true;
        }

        return false;
    }

    public function attachToContent(Content $content, ?string $author = null): ?Proposal
    {
        if (! $author) {
            $author = $this->getContentAuthorUuid($content);
        }

        $author = User::find($author);

        if (! $author) {
            Log::error('Content author does not exist');

            return null;
        }

        $authorRoles = Role::listUserRoleIds($author);
        /** @var Collection<Workflow> $workflows */
        $workflows = Workflow::where('active', 1)->get();
        $data = [
            'workflow_id' => null,
            'proposable_uuid' => $content->uuid,
            'proposable_type' => get_class($content),
        ];

        foreach ($workflows as $workflow) {
            // We check if user roles match with the workflow contributeurs
            if (empty(array_intersect($authorRoles, $workflow->contributors()->pluck('inside_roles.id')->toArray()))) {
                continue;
            }

            $this->checkConditions($data, $workflow, $content);

            if ($data['workflow_id']) {
                break;
            }
        }

        if (! $data['workflow_id']) {
            return null;
        }

        $proposal = Proposal::firstOrCreate($data);
        $this->addCycle($proposal->id);

        return $proposal;
    }

    protected function checkConditions(array &$data, Workflow $workflow, Content $content): void
    {
        if (count($workflow->contents) === 0) {
            return;
        }

        foreach ($workflow->contents as $type) {
            if (class_to_type($content) !== $type) {
                continue;
            }

            if (empty($workflow->conditions)) {
                $data['workflow_id'] = $workflow->id;
                continue;
            }

            foreach ($workflow->conditions as $condition => $value) {
                [$field, $operator] = explode(':', $condition);

                if (! Schema::hasField(class_to_type($content), $field)) {
                    return;
                }

                switch ($operator) {
                    case 'eq':
                        if ($content->{$field} !== $value) {
                            return;
                        }

                        break;
                    case 'in':
                        $fieldOptions = Schema::getFieldOptions(class_to_type($content), $field);

                        switch ($fieldOptions['type']) {
                            case 'reference':
                                $target = $fieldOptions['target'][0];
                                $uuidHosts = DB::table(type_to_table($target))->whereIn('uuid', $value)->pluck('uuid_host');
                                $uuids = DB::table(type_to_table($target))->whereIn('uuid_host', $uuidHosts)->pluck('uuid');

                                $hasReference = DB::table('inside_pivots')->where('parent_uuid', $content->uuid)->where('related_type', type_to_class($fieldOptions['target'][0]))->whereIn('related_uuid', $uuids)->exists();

                                if (! $hasReference) {
                                    return;
                                }

                                break;
                        }
                        break;
                }
            }

            $data['workflow_id'] = $workflow->id;
        }
    }

    public function isContributor(Proposal $proposal, User $user): bool
    {
        $userRoles = Role::listUserRoleIds($user);

        if (empty(array_intersect($userRoles, $proposal->workflow->contributors()->pluck('inside_roles.id')->toArray()))) {
            return false;
        }

        return true;
    }

    public function isReviewer(Proposal $proposal, User $user): bool
    {
        foreach ($proposal->workflow->steps as $step) {
            if ($this->isStepReviewer($step, $proposal, $user)) {
                return true;
            }
        }

        return false;
    }

    public function isStepReviewer(ProposalStep|WorkflowStep $step, Proposal $proposal, User $user): bool
    {
        if ($user->hasAnyRole('super_administrator')) {
            return true;
        }

        $workflowStep = $step instanceof WorkflowStep ? $step : $step->workflowStep;
        $userRoles = Role::listUserRoleIds($user);

        if ($workflowStep->step_type === 'designation') {
            return $user->uuid === \Inside\Workflow\Facades\Proposal::getContentAuthorUuid($proposal->getProposableModelAttribute());
        }

        if ($workflowStep->step_type === 'validation' && $step instanceof ProposalStep) {
            $previousStep = $this->getPreviousStep($step->cycle->proposal->id, $step->id);

            if ($previousStep && $previousStep->workflowStep->step_type === 'designation') {
                return $user->uuid === $previousStep->reviewer_uuid;
            }
        }

        if (array_intersect($userRoles, $workflowStep->reviewers()->pluck('inside_roles.id')->toArray())) {
            return true;
        }

        return false;
    }

    public function getFromContent(string $uuid, string $type): ?Proposal
    {
        return Proposal::where([
            'proposable_uuid' => $uuid,
            'proposable_type' => $type,
        ])
        ->first();
    }

    public function addCycle(int $id): ?Proposal
    {
        $proposal = Proposal::find($id);
        if (! $proposal) {
            return null;
        }

        $proposalCycle = $proposal->cycles()->create();

        // We fill the cycle with the steps defined in the workflow
        $workflowSteps = $proposal->workflow->steps;
        foreach ($workflowSteps as $workflowStep) {
            $proposalCycle->steps()->create([
                'step_id' => $workflowStep->id,
                'validated' => false,
                'review' => '',
            ]);
        }

        // The proposal come back to the "in review" status
        $proposal->status = 2;
        $proposal->save();

        return $proposal;
    }

    public function isLastStep(int $id): bool
    {
        $step = ProposalStep::find($id);
        if (! $step) {
            return true;
        }
        $lastStep = $step->cycle->steps()->latest()->orderBy('id', 'desc')->first();

        return $step->id === $lastStep->id;
    }

    public function isFirstStep(int $id): bool
    {
        $step = ProposalStep::find($id);
        if (! $step) {
            return false;
        }
        $firstStep = $step->cycle->steps()->latest()->orderBy('id', 'asc')->first();

        return $step->id === $firstStep->id;
    }

    public function getPreviousStep(int $id, ?int $stepId = null): ?ProposalStep
    {
        $proposal = Proposal::find($id);

        if ($proposal) {
            if ($stepId) {
                $currentStep = ProposalStep::find($stepId);
                if (! $currentStep) {
                    return null;
                }
                $lastCycle = $currentStep->cycle;
            } else {
                $lastCycle = $proposal->cycles()->latest()->orderBy('id', 'desc')->first();
                $currentStep = $lastCycle->steps()->whereNotNull('user_uuid')->latest()->orderBy('id', 'desc')->first();
            }

            if ($currentStep) {
                return $lastCycle->steps()->where('id', '<', $currentStep->id)->orderByDesc('id')->first();
            } else {
                return $lastCycle->steps()->orderBy('id')->first();
            }
        }

        return null;
    }

    public function getCurrentStep(int $id, bool|int $stepId = false): ?ProposalStep
    {
        $proposal = Proposal::find($id);

        if (! is_null($proposal)) {
            $lastCycle = $proposal->cycles()->latest()->orderBy('id', 'desc')->first();

            if ($stepId) {
                return $lastCycle->steps()->where('step_id', $stepId)->first();
            } else {
                return $lastCycle->steps()->whereNotNull('user_uuid')->latest()->orderBy('id', 'desc')->first();
            }
        }

        return null;
    }

    public function getNextStep(int $id): ?ProposalStep
    {
        $proposal = Proposal::find($id);

        if (! $proposal) {
            return null;
        }

        $lastCycle = $proposal->cycles()->latest()->orderBy('id', 'desc')->first();
        $currentStep = $lastCycle->steps()->whereNotNull('user_uuid')->latest()->orderBy('id', 'desc')->first();

        if ($currentStep) {
            return $lastCycle->steps()->where('id', '>', $currentStep->id)->orderBy('id')->first();
        }

        return $lastCycle->steps()->orderBy('id')->first();
    }

    public function isContentDeclined(Content $content): bool
    {
        $customCallback = config('workflow.content_is_declined', false);

        if ($customCallback && is_callable($customCallback) && $customCallback($content)) {
            return true;
        }

        return false;
    }

    /**
     * @throws Exception
     */
    public function editStep(int $id, array $data, bool $stopPropagation = false): ?Proposal
    {
        $step = ProposalStep::find($id);
        if (! $step) {
            return null;
        }
        $proposal = $step->cycle->proposal;
        $proposable = $proposal->getProposableModelAttribute();

        if ($step->workflowStep->step_type === 'designation' && array_key_exists('user_uuid', $data) && ! empty($data['user_uuid'])) {
            $data['user_uuid'] = $proposable->update_author;
        }

        $step->update($data);
        $reviewed = false;

        if ($stopPropagation) {
            return $proposal;
        }

        if ($this->isLastStep($id)) {
            $reviewed = true;
            $proposal->status = (int) $step->validated;
            $proposal->save();

            if ($step->validated) {
                $this->editContentStatus($proposable, true);
            }
        } else {
            $proposal->status = $step->validated ? 2 : 0;

            if (! $step->validated) {
                $reviewed = true;
            }
            $proposal->save();
        }
        if ($reviewed) {
            ProposalReviewedEvent::dispatch($proposal, $proposable);
        } else {
            ProposalEditedEvent::dispatch($proposal, $proposable);
        }

        return $proposal;
    }

    /**
     * @throws Exception
     */
    public function editContentStatus(Content $content, bool $status): void
    {
        $bridge = new BridgeContent;
        $bridge->setStatus($content, $status);
        $this->forgetCache($content);
        if (Package::has('lhg-commons')) {
            foreach (list_languages() as $langcode) {
                if ($langcode === $content->langcode) {
                    continue;
                }
                $translatedContent = $content->getTranslationIfExists($langcode);
                $bridge->setStatus($translatedContent, $status);
                $this->forgetCache($translatedContent);
            }
        }
    }

    /**
     * @param mixed $model
     * @return void
     */
    protected function forgetCache($model)
    {
        $tags = ContentCache::getTags($model->content_type);
        $allowedFieldTypes = ['reference', 'comments', 'reactions'];
        $modelType = class_to_type($model);
        try {
            $fields = Schema::getFieldListing($modelType);
        } catch (FieldSchemaNotFoundException $e) {
            $fields = [];
        }

        foreach ($fields as $fieldName) {
            $fieldOptions = Schema::getFieldOptions($modelType, $fieldName);
            if (in_array($fieldOptions['type'], $allowedFieldTypes)) {
                $tags = array_merge($tags, (array) $fieldOptions['target']);
            }
        }
        ContentCache::forget($tags);
    }

    /**
     * @param int $id
     * @return array|null
     */
    public function getHistories(int $id): ?array
    {
        $histories = [];
        $proposal = Proposal::find($id);

        if (! $proposal) {
            return null;
        }

        foreach ($proposal->cycles as $cycleKey => $cycle) {
            $histories[$cycleKey] = [];

            foreach ($cycle->steps as $step) {
                $author = call_user_func('Inside\Content\Models\Contents\Users::query')->find($step->user_uuid);
                $history = [
                    'id' => $step->id,
                    'type' => $step->workflowStep->step_type,
                    'author' => [
                        'uuid' => $author ? $author->uuid : null,
                        'image' => $author ? protected_file_url($author, 'image') : null,
                        'firstname' => $author ? $author->firstname : null,
                        'lastname' => $author ? $author->lastname : null,
                    ],
                    'created_at' => $step->updated_at,
                    'review' => $step->review,
                    'validated' => $step->validated,
                    'isReviewer' => $this->isStepReviewer($step, $proposal, Permission::user()),
                ];

                if ($step->workflowStep->step_type === 'designation') {
                    $history['roles'] = [];

                    foreach ($step->workflowStep->reviewers as $group) {
                        $history['roles'][] = $group->id;
                    }

                    if ($step->validated) {
                        $reviewer = call_user_func('Inside\Content\Models\Contents\Users::query')->find($step->reviewer_uuid);
                        $history['reviewer'] = [
                            'uuid' => $reviewer ? $reviewer->uuid : null,
                            'image' => $reviewer ? protected_file_url($reviewer, 'image') : null,
                            'firstname' => $reviewer ? $reviewer->firstname : null,
                            'lastname' => $reviewer ? $reviewer->lastname : null,
                        ];
                    }
                }
                $histories[$cycleKey][] = $history;
            }
        }

        return $histories;
    }
}
