<?php

declare(strict_types=1);

namespace Inside\Workflow\Listeners;

use Exception;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Facades\Role;
use Inside\Workflow\Events\ProposalEditedEvent;
use Inside\Workflow\Facades\Proposal;
use Inside\Workflow\Facades\Workflow;
use Inside\Workflow\Models\Proposal as ProposalModel;
use Inside\Workflow\Models\ProposalCycle;
use Inside\Workflow\Services\ProposalService;

final class ProposalEditedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ProposalEditedEvent $event): void
    {
        if (! Workflow::isWorkflowEnable()) {
            return;
        }
        $model = $event->proposable;
        $this->sendNotifications($event->proposal, $model, $event->steps);
        $languages = list_languages();

        if (count($languages) > 1 && config('workflow.multilingual_type') === 'automatic') {
            foreach ($languages as $language) {
                if ($model->langcode === $language) {
                    continue;
                }

                $this->handleTranslation($model, $language, $event->steps);
            }
        }
    }

    /**
     * @throws Exception
     */
    protected function handleTranslation(Content $model, string $langcode, array $steps): void
    {
        $translation = $model->getTranslationIfExists($langcode);

        if ($translation->langcode === $model->langcode || is_null($model->uuid) || is_null($translation->uuid)) {
            return;
        }

        // Get proposal from translation
        $proposal = Proposal::getFromContent($translation->uuid, get_class($translation));

        if (is_null($proposal)) {
            return;
        }

        $service = new ProposalService();
        $originalProposal = Proposal::getFromContent($model->uuid, get_class($model));
        if (is_null($originalProposal)) {
            return;
        }

        $proposal->cycles()->delete();

        /** @var ProposalCycle $cycle */
        foreach ($originalProposal->cycles()->get() as $cycle) {
            $service->addCycle($proposal->id);

            foreach ($cycle->steps()->get() as $originalStep) {
                $step = Proposal::getCurrentStep($proposal->id, $originalStep->step_id);

                foreach (['previous', 'current', 'next'] as $stepName) {
                    if ($steps[$stepName] && $steps[$stepName]->id === $step->id) {
                        $steps[$stepName] = $step;
                    }
                }

                $service->editStep($step->id, [
                    'review' => $originalStep->review,
                    'reviewer_uuid' => $originalStep->reviewer_uuid,
                    'user_uuid' => $originalStep->user_uuid,
                    'validated' => $originalStep->validated,
                ], true);

                DB::table('inside_workflow_proposal_steps')->where('id', $step->id)->update(
                    [
                        'created_at' => date('Y-m-d H:i:s', $originalStep->created_at),
                        'updated_at' => date('Y-m-d H:i:s', $originalStep->updated_at),
                    ]
                );
            }
        }

        $originalProposal = Proposal::getFromContent($model->uuid, get_class($model));
        if (is_null($originalProposal)) {
            return;
        }

        $proposal->status = $originalProposal->status;
        $proposal->save();

        $status = $proposal->status === 1;
        $service->editContentStatus($translation, $status);

        $this->sendNotifications($proposal, $translation, $steps);
    }

    /**
     * @param  ProposalModel  $proposal
     * @param  mixed  $proposable
     * @param  array  $steps
     * @return void
     */
    protected function sendNotifications(ProposalModel $proposal, $proposable, array $steps)
    {
        $nextStep = $steps['next'];

        if (is_null($nextStep)) {
            return;
        }

        // Get the next step global configuration (reviewers, alert types...)
        $workflowNextStep = $nextStep->workflowStep;
        $reviewers = [];

        switch ($workflowNextStep->step_type) {
            case 'validation':
                $currentStep = $steps['current'];
                $workflowCurrentStep = ! empty($currentStep) ? $currentStep->workflowStep : false;

                if (! $workflowCurrentStep || $workflowCurrentStep->step_type === 'validation') {
                    foreach ($workflowNextStep->reviewers as $group) {
                        $roles = Role::listRoleUsers($group->id, null, null, true);

                        foreach ($roles['data'] as $user) {
                            $reviewers[] = $user->uuid;
                        }
                    }
                } elseif ($workflowCurrentStep->step_type === 'designation') {
                    $reviewers[] = $currentStep->reviewer_uuid;
                }
                break;
            case 'designation':
                $reviewers[] = Proposal::getContentAuthorUuid($proposable);
                break;
        }

        $subscribers = User::whereIn('uuid', $reviewers)->get();

        /** @var NotificationType $type */
        $type = NotificationType::where([
            'event' => 'Inside\Workflow\Events\ProposalEdited',
            'action' => 'edit',
            'via' => 'web',
        ])->first();

        foreach ($subscribers as $subscriber) {
            $this->route = $subscriber;

            $steps = [
                'current' => (isset($workflowCurrentStep) && ! empty($workflowCurrentStep)) ? $workflowCurrentStep->step_type : null,
                'next' => $workflowNextStep->step_type,
            ];

            $md5Data = [
                'uuid' => $proposable->uuid,
                'subscriber' => $subscriber->uuid,
                'steps' => $steps,
                'cycles' => $proposal->cycles->count(),
            ];

            $md5 = md5((string) json_encode($md5Data));

            if (! $this->checkConditions($type, $proposal, $subscriber) || ! $this->checkUniqueness($type, $proposal,
                $subscriber)) {
                continue;
            }

            if (Notification::query()->where('data', 'LIKE', '%"md5":"'.$md5.'"%')->exists()) {
                continue;
            }

            $data = [
                'url' => 'edit/'.class_to_type($proposal->proposable_type).'/'.$proposal->proposable_uuid,
                'steps' => $steps,
                'sendMail' => $workflowNextStep->send_mail,
                'md5' => $md5,
            ];

            $customNotifications = config('workflow.custom_notifications', false);

            if ($customNotifications && is_callable($customNotifications)) {
                $customNotifications($data, $proposable, 'web', 'edit');
            }

            $this->when = Carbon::now();
            $this->notify($type, $subscriber, $proposable, $data);
        }
    }
}
