<?php

declare(strict_types=1);

namespace Inside\Workflow\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Content\Models\Model;
use Inside\Notify\Events\NotificationCreatedEvent;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationType;
use Inside\Workflow\Facades\Workflow;
use Inside\Workflow\Models\Proposal;

final class NotificationCreatedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(NotificationCreatedEvent $event): void
    {
        if (! Workflow::isWorkflowEnable()) {
            return;
        }
        $notification = $event->notification;
        $notificationType = $notification->trigger;

        if ($notificationType->via !== 'web' || $notificationType->model !== Proposal::class) {
            return;
        }

        if (! array_key_exists('sendMail', $notification->data) || ! $notification->data['sendMail']) {
            return;
        }

        /** @var Content $model */
        $model = $notification->notifiable;
        $multilingual = config('workflow.multilingual_mail_notifications', true);
        /** @var ?User $to */
        $to = User::find($notification->user_uuid);
        if (is_null($to)) {
            return;
        }

        if (! $multilingual && $model->langcode !== $to->langcode) {
            return;
        }

        /** @var NotificationType $emailNotificationType */
        $emailNotificationType = NotificationType::where([
            'event'  => $notificationType->event,
            'action' => $notificationType->action,
            'via' => 'email',
        ])->first();
        $data = $notification->data;
        $data['extra']['mail'] = data_get($emailNotificationType->data, 'mail', []);
        $customNotifications = config('workflow.custom_notifications', false);

        if ($customNotifications && is_callable($customNotifications)) {
            $customNotifications($data, $model, 'email', $notificationType->action);
        }

        if (is_null($model->uuid)) {
            return;
        }

        $proposal = \Inside\Workflow\Facades\Proposal::getFromContent($model->uuid, get_class($model));
        if (is_null($proposal)) {
            return;
        }
        $step = \Inside\Workflow\Facades\Proposal::getCurrentStep($proposal->id);

        if ($step && ! is_null($step->user_uuid)) {
            $data['extra']['mail']['subtext'] = Lang::get('notifications.workflow.mail.comment', [], $to->langcode).strip_tags($step->review);
        }

        $this->route = $to;
        $this->when = Carbon::now();
        $this->notify($emailNotificationType, $to, $model, $data);
    }
}
