<?php

namespace Inside\Workflow\Http\Resources;

use Illuminate\Http\Resources\Json\Resource;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Workflow\Models\Workflow;

/**
 * @mixin Workflow
 */
class WorkflowResource extends Resource
{
    public function toArray($request): array
    {
        if ($request->is('api/v1/workflow/*/in-progress')) {
            return $this->getRelatedContents(2);
        }

        if ($request->is('api/v1/workflow/*/done')) {
            return $this->getRelatedContents(0);
        }

        $currentUser = Auth::user();
        $conditions = $this->conditions ?? [];

        foreach ($conditions as $condition => &$value) {
            [$field, $operator] = explode(':', $condition);

            foreach ($this->contents as $contentType) {
                if (! Schema::hasField($contentType, $field)) {
                    continue;
                }

                $fieldOptions = Schema::getFieldOptions($contentType, $field);

                if ($fieldOptions['type'] !== 'reference') {
                    continue;
                }

                foreach ($value as $key => &$uuid) {
                    $content = call_user_func(type_to_class($fieldOptions['target'][0]).'::find', $uuid);

                    if (! $content || $content->langcode === $currentUser->langcode) {
                        continue;
                    }

                    $translation = $content->getTranslationIfExist($currentUser->langcode);

                    if ($translation->langcode !== $content->langcode) {
                        $uuid = $translation->uuid;
                    }
                }
            }
        }

        $contents = [];

        foreach ($this->contents as $content) {
            $modelOptions = Schema::getModelOptions($content);
            $contents[] = [
                'content_type' => $content,
                'title' => $modelOptions['title'],
            ];
        }

        return [
            'id' => $this->id,
            'title' => $this->title,
            'active' => $this->active,
            'steps' => WorkflowStepResource::collection($this->steps),
            'contributors' => $this->contributors->map->only(['id', 'name']),
            'notified_roles' => $this->notified_roles->map->only(['id', 'name']),
            'send_mail' => $this->send_mail,
            'contents' => $contents,
            'conditions' => $conditions,
            'related_contents_published_count' => $this->proposals()->where('status', 1)->count(),
            'related_contents_declined_count' => $this->proposals()->where('status', 0)->count(),
            'related_contents_in_progress_count' => $this->proposals()->where('status', 2)->count(),
        ];
    }

    public function getRelatedContents(int $status): array
    {
        $types = $this->proposals()
            ->distinct()
            ->where(
                function ($query) use ($status) {
                    $query->where('status', $status);
                    if ($status == 0) {
                        $query->orWhere('status', 1);
                    }
                }
            )
            ->pluck('proposable_type');

        $proposals = $this->proposals()
            ->select('inside_workflow_proposals.*')
            ->where(
                function ($query) use ($status) {
                    $query->where('status', $status);
                    if ($status == 0) {
                        $query->orWhere('status', 1);
                    }
                }
            )
            ->where(function ($query) use ($types) {
                $types->map(function ($type) use ($query) {
                    $query->orWhere(function ($subquery) use ($type) {
                        $subquery->whereExists(function ($subsubquery) use ($type) {
                            $subsubquery->select(DB::raw(1))
                                ->from(class_to_table($type))
                                ->whereColumn('uuid', 'inside_workflow_proposals.proposable_uuid');
                        });
                    });
                });
            })
            ->orderBy('id', 'desc')
            ->paginate(6);

        $relatedContents = (ProposalResource::collection($proposals))->resolve();

        return [
            'data' => $relatedContents,
            'current_page' => $proposals->currentPage(),
            'last_page' => $proposals->lastpage(),
            'per_page' => $proposals->perPage(),
            'total' => $proposals->total(),
        ];
    }
}
