<?php

namespace Inside\Workflow\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Workflow\Facades\Proposal;
use Symfony\Component\HttpFoundation\Response;

/**
 * Restrict form query
 */
class WorkflowContentFormMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $method = $request->getMethod();
        $path = $request->path();

        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        // Check path
        if (! strpos($path, 'api/v1/form/') == 0) {
            return $response;
        }

        preg_match('/^api\/v1\/form\/(.*)?$/', $path, $match);

        if (! isset($match[1])) {
            return $response;
        }

        $exploded = explode('/', $match[1]);
        $uuid = null;

        if (isset($exploded[1])) {
            $match[1] = $exploded[0];
            $uuid = $exploded[1];
        }

        $class = type_to_class($match[1]);

        if (! $uuid) {
            return $response;
        }

        $content = call_user_func($class.'::find', $uuid);

        return $this->addProposalToForm($response, $content);
    }

    /**
     * @param Response $response
     * @param Content|null $content
     * @return Response
     */
    public function addProposalToForm(Response $response, ?Content $content)
    {
        if (! $content) {
            return $response;
        }
        $data = json_decode_response($response);

        if (empty($data) || is_null($content->uuid)) {
            return $response;
        }

        $proposal = Proposal::getFromContent($content->uuid, get_class($content));
        $user = Auth::user();

        if (! $proposal || ! $user) {
            return $response;
        }
        // If the proposal is in review and the current user is not a reviewer, we do not display the workflow block
        if ($proposal->status === 2 && ! Proposal::isReviewer($proposal, User::find($user->uuid))) {
            return $response;
        }

        $data['data'][] = [
            'id' => 'group_workflow',
            'type' => 'fieldset',
            'label' => [
                'en' => 'Workflow',
                'fr' => 'Workflow',
            ],
            'fields' => [
                [
                    'name' => 'workflow',
                    'options' => [
                        'id' => $proposal->id,
                        'status' => $proposal->status,
                        'histories' => Proposal::getHistories($proposal->id),
                    ],
                    'type' => 'workflow',
                    'weight' => 0,
                ],
            ],
        ];

        set_response($response, $data);

        return $response;
    }
}
