<?php

declare(strict_types=1);

namespace Inside\Workflow\Services;

use Closure;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;
use Inside\Permission\Models\Role as RoleModel;
use Inside\Workflow\Contracts\Workflow as WorkflowContract;
use Inside\Workflow\Models\Workflow;

final class WorkflowService implements WorkflowContract
{
    protected bool $enable = true;

    protected ?array $conditions = null;

    private function createRoleGroup(Workflow $workflow): void
    {
        $workflowRole = Role::create([
            'name' => 'workflow-'.$workflow->id,
            'type' => 'workflow',
        ]);

        foreach ($workflow->contents as $content) {
            $this->handleWorkflowPermission($content, $workflowRole);
        }

        $conditions = $this->conditions ?? [];
        $categories = [];

        foreach ($conditions as $condition => $value) {
            [$field, $operator] = explode(':', $condition);

            foreach ($workflow->contents as $contentType) {
                if (! Schema::hasField($contentType, $field)) {
                    continue;
                }
                $fieldOptions = Schema::getFieldOptions($contentType, $field);

                if ($fieldOptions['type'] !== 'reference') {
                    continue;
                }

                foreach ($value as $key => $uuid) {
                    $categories[] = [
                        'type' => $fieldOptions['target'][0],
                        'uuid' => $uuid,
                    ];
                    $contentSource = DB::table(type_to_table($fieldOptions['target'][0]))->where('uuid', $uuid)->first();

                    if (! isset($contentSource->uuid_host, $contentSource->langcode)) {
                        continue;
                    }

                    $translations = DB::table(type_to_table($fieldOptions['target'][0]))->where('uuid_host', $contentSource->uuid_host)->where('langcode', '<>', $contentSource->langcode)->get();

                    foreach ($translations as $translation) {
                        $categories[] = [
                            'type' => $fieldOptions['target'][0],
                            'uuid' => $translation->uuid,
                        ];
                    }
                }
                continue 2;
            }
        }

        foreach ($categories as $category) {
            $this->handleWorkflowPermission($category, $workflowRole);
        }

        foreach ($workflow->contributors as $contributor) {
            $usersUUID = Role::listRoleUsers($contributor->id, null, null, true)['data']->pluck('uuid')->toArray();
            Role::assignUsersToRole($workflowRole->id, $usersUUID);
        }
    }

    private function deleteRoleGroup(?Workflow $workflow): void
    {
        if (! $workflow) {
            return;
        }

        /** @var ?RoleModel $workflowRole */
        $workflowRole = RoleModel::where([
            'name' => 'workflow-'.$workflow->id,
        ])->first();

        if ($workflowRole) {
            Role::delete($workflowRole->id);
        }
    }

    private function updateRoleGroup(Workflow $workflow): void
    {
        /** @var ?RoleModel $workflowRole */
        $workflowRole = RoleModel::where([
            'name' => 'workflow-'.$workflow->id,
        ])->first();

        if ($workflowRole) {
            DB::table('inside_users_roles')->where('role_id', $workflowRole->id)->delete();

            foreach ($workflow->contributors as $contributor) {
                $usersUUID = Role::listRoleUsers($contributor->id, null, null, true)['data']->pluck('uuid')->toArray();
                Role::assignUsersToRole($workflowRole->id, $usersUUID);
            }
        }
    }

    /**
     * @param array<string, string>|string $content
     * @param RoleModel $role
     * @return void
     */
    private function handleWorkflowPermission(array|string $content, RoleModel $role): void
    {
        $this->generatePermission($content, $role->id, 'read', 'role_id');
        $this->generatePermission($content, $role->id, 'create', 'role_id');
    }

    /**
     * @param array<string, string>|string $content
     * @param int $id
     * @param string $action
     * @param string $attribute
     * @return void
     */
    private function generatePermission(array|string $content, int $id, string $action = 'read', string $attribute = 'role_id'): void
    {
        if (isset($content['type'])) {
            $class = type_to_class($content['type']);
        } elseif (is_string($content)) {
            $class = type_to_class($content);
        } else {
            $class = Arr::first($content);
        }

        $data = [
            'authorizable_type' => $class,
            'authorizable_uuid' => null,
            'action'            => $action,
            'invert'            => false,
            'children'          => true,
        ];

        if (isset($content['uuid'])) {
            $data['authorizable_uuid'] = $content['uuid'];
        }

        $permissionSchemaID = DB::table('inside_permissions_schema')->where($data)->pluck('id')->first();

        if (! $permissionSchemaID) {
            $permissionSchemaID = DB::table('inside_permissions_schema')->insertGetId($data);
        }

        if ($attribute == 'role_id') {
            $schema = [
                'role_id'              => $id,
                'permission_schema_id' => $permissionSchemaID,
            ];

            if (! DB::table('inside_roles_permissions_schema')->where($schema)->exists()) {
                DB::table('inside_roles_permissions_schema')->insert($schema);
            }
        }
        Permission::createPermission($id, $data, $attribute);
    }

    public function list(): Collection
    {
        return Workflow::with('steps')->get(); // @phpstan-ignore-line
    }

    public function get(int $id): ?Workflow
    {
        return Workflow::with('steps')->find($id); // @phpstan-ignore-line
    }

    public function create(array $data): Workflow
    {
        $workflow = Workflow::create($data);
        $workflow->contributors()->attach($data['contributors']);

        if (array_key_exists('notified_roles', $data)) {
            $workflow->notified_roles()->attach($data['notified_roles']);
        }

        foreach ($data['steps'] as $key => $stepData) {
            $step = $workflow->steps()->create($stepData);
            $step->reviewers()->attach($stepData['reviewers']);
        }
        $this->createRoleGroup($workflow);

        return $workflow;
    }

    public function update(int $id, array $data): ?Workflow
    {
        $workflow = Workflow::find($id);

        if (! $workflow) {
            return null;
        }

        $workflow->update($data);
        $workflow->notified_roles()->sync($data['notified_roles']);
        $steps = [];

        foreach ($data['steps'] as $stepData) {
            $steps[$stepData['id']] = [
                'title' => $stepData['title'],
                'send_mail' => $stepData['send_mail'],
            ];
        }

        foreach ($workflow->steps as $step) {
            if (! array_key_exists($step->id, $steps)) {
                $step->delete();
                continue;
            }

            $step->update($steps[$step->id]);
        }

        foreach ($data['steps'] as $stepData) {
            if ($workflow->steps->find($stepData['id'])) {
                continue;
            }
            $step = $workflow->steps()->create($stepData);
            $step->reviewers()->sync($stepData['reviewers']);
        }
        $this->updateRoleGroup($workflow);

        return $workflow;
    }

    public function delete(int $id): void
    {
        $workflow = Workflow::find($id);
        $this->deleteRoleGroup($workflow);
        try {
            $workflow?->delete();
        } catch (Exception $exception) {
            $message = __(
                '[WorkflowService::delete] failed to delete workflow :id => $message',
                [
                    'id' => $id,
                    'message' => $exception->getMessage(),
                ]
            );

            if (is_string($message)) {
                Log::error($message);
            }
        }
    }

    public function isContentInWorkflow(array|string $contentTypes): bool
    {
        if (empty($contentTypes)) {
            return false;
        }

        if (is_string($contentTypes)) {
            $contentTypes = [$contentTypes];
        }

        $query = Workflow::query();
        foreach ($contentTypes as $contentType) {
            $query->orWhere('contents', 'like', "%\"$contentType\"%");
        }

        return $query->exists();
    }

    public function withoutWorkflow(Closure $callback): mixed
    {
        $this->disableWorkflow();

        try {
            return $callback();
        } finally {
            $this->enableWorkflow();
        }
    }

    public function isWorkflowEnable(): bool
    {
        return $this->enable;
    }

    public function enableWorkflow(): void
    {
        $this->enable = true;
    }

    public function disableWorkflow(): void
    {
        $this->enable = false;
    }
}
