<?php

declare(strict_types=1);

namespace Inside\Workflow\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Carbon;
use Inside\Permission\Models\Role;

/**
 * @property int $id
 * @property bool $active
 * @property string $title
 * @property array $contents
 * @property array|null $conditions
 * @property bool $send_mail
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Collection<Role> $contributors
 * @property-read int $contributors_count
 * @property Collection<Role> $notified_roles
 * @property-read int $notified_roles_count
 * @property Collection<Proposal> $proposals
 * @property-read int $proposals_count
 * @property Collection<WorkflowStep> $steps
 * @property-read int $steps_count
 *
 * @method static Collection|Workflow[] get(array $columns = ['*'])
 * @method static null|Workflow find($id, $columns = ['*'])
 * @method static Builder where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Workflow create(array $attributes = [])
 * @method static Builder|Workflow newModelQuery()
 * @method Builder|Workflow newQuery()
 * @method static Builder|Workflow query()
 * @method static Builder|Workflow with(array|string $relations)
 *
 * @foreignLinks id,\Inside\Workflow\Models\WorkflowStep,workflow_id|id,\Inside\Workflow\Models\Proposal,workflow_id
 */
final class Workflow extends Model
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inside_workflows';

    /**
     * Casts the following attributes
     *
     * @var array
     */
    protected $casts = [
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
        'contents' => 'array',
        'conditions' => 'array',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'active',
        'contents',
        'conditions',
        'send_mail',
    ];

    /**
     * @return HasMany
     */
    public function steps(): HasMany
    {
        return $this->hasMany(WorkflowStep::class);
    }

    /**
     * Get all contributors roles.
     *
     * @return BelongsToMany
     */
    public function contributors(): BelongsToMany
    {
        return $this->belongsToMany(
            Role::class,
            'inside_contributors_workflows',
            'workflow_id',
            'role_id'
        );
    }

    /**
     * Get all notified roles.
     *
     * @return BelongsToMany
     */
    public function notified_roles(): BelongsToMany
    {
        return $this->belongsToMany(
            Role::class,
            'inside_notified_roles_workflows',
            'workflow_id',
            'role_id'
        );
    }

    /**
     * Get all proposals.
     *
     * @return HasMany
     */
    public function proposals(): HasMany
    {
        return $this->hasMany(
            Proposal::class,
            'workflow_id'
        );
    }
}
