<?php

declare(strict_types=1);

namespace Inside\Workflow\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Models\Role;
use Inside\Workflow\Events\ProposalReviewedEvent;
use Inside\Workflow\Facades\Proposal;
use Inside\Workflow\Facades\Workflow;
use Inside\Workflow\Models\Proposal as ProposalModel;
use Inside\Workflow\Models\ProposalCycle;
use Inside\Workflow\Models\ProposalStep;
use Inside\Workflow\Services\ProposalService;

final class ProposalReviewedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ProposalReviewedEvent $event): void
    {
        if (! Workflow::isWorkflowEnable()) {
            return;
        }
        $model = $event->proposable;

        $this->sendNotifications($event->proposal, $event->proposable, $event->steps);

        $languages = list_languages();

        if (count($languages) > 1 && config('workflow.multilingual_type') === 'automatic') {
            foreach ($languages as $language) {
                if ($model->langcode === $language) {
                    continue;
                }
                $this->handleTranslation($model, $language, $event->steps);
            }
        }
    }

    protected function handleTranslation(Content $model, string $langcode, array $steps): void
    {
        $translation = $model->getTranslationIfExists($langcode);

        if ($translation->langcode === $model->langcode || is_null($model->uuid) || is_null($translation->uuid)) {
            return;
        }

        // Get proposal from translation
        $proposal = Proposal::getFromContent($translation->uuid, get_class($translation));

        if (is_null($proposal)) {
            return;
        }

        $service = new ProposalService();
        $originalProposal = Proposal::getFromContent($model->uuid, get_class($model));
        if (is_null($originalProposal)) {
            return;
        }
        $proposal->cycles()->delete();

        /** @var ProposalCycle $cycle */
        foreach ($originalProposal->cycles()->get() as $cycle) {
            $service->addCycle($proposal->id);

            foreach ($cycle->steps()->get() as $originalStep) {
                $step = Proposal::getCurrentStep($proposal->id, $originalStep->step_id);

                foreach (['previous', 'current', 'next'] as $stepName) {
                    if ($steps[$stepName] && $steps[$stepName]->id === $step->id) {
                        $steps[$stepName] = $step;
                    }
                }
                $service->editStep($step->id, [
                    'review' => $originalStep->review,
                    'reviewer_uuid' => $originalStep->reviewer_uuid,
                    'user_uuid' => $originalStep->user_uuid,
                    'validated' => $originalStep->validated,
                ], true);

                DB::table('inside_workflow_proposal_steps')->where('id', $step->id)->update(
                    [
                        'created_at' => date('Y-m-d H:i:s', $originalStep->created_at),
                        'updated_at' => date('Y-m-d H:i:s', $originalStep->updated_at),
                    ]
                );
            }
        }

        $originalProposal = Proposal::getFromContent($model->uuid, get_class($model));
        if (is_null($originalProposal)) {
            return;
        }
        $proposal->status = $originalProposal->status;
        $proposal->save();
        $status = $proposal->status === 1;
        $service->editContentStatus($translation, $status);
        $this->sendNotifications($proposal, $translation, $steps);
    }

    protected function sendNotifications(ProposalModel $proposal, Content $proposable, array $steps): void
    {
        // If the proposal is not accepted nor declined, we do nothing
        if ($proposal->status !== 1 && $proposal->status !== 0) {
            return;
        }

        /** @var ProposalStep $currentStep */
        $currentStep = $steps['current'];

        /** @var ?ProposalStep $previousStep */
        $previousStep = $steps['previous'];

        // Get the current step global configuration (reviewers, alert types...)
        $workflowCurrentStep = $currentStep->workflowStep;
        $workflowPreviousStep = $previousStep?->workflowStep ?? null;

        $steps = [
            'previous' => $workflowPreviousStep?->step_type ?? null,
            'current' => $workflowCurrentStep->step_type,
        ];

        /** @var User $author */
        $author = User::find(Proposal::getContentAuthorUuid($proposable));

        $subscribers = $proposal->workflow->notified_roles
            ->flatMap(fn (Role $role) => $role->users)
            ->where('status', 1)
            ->push($author)
            ->unique('uuid')
            ->reject(fn (User $user) => $user->uuid === $currentStep->user_uuid);

        // If the proposal is accepted or declined we send a notification to the contributor
        $notificationAction = ($proposal->status === 1) ? 'accept' : 'decline';

        /** @var NotificationType $type */
        $type = NotificationType::where([
            'event' => 'Inside\Workflow\Events\ProposalReviewed',
            'action' => $notificationAction,
            'via' => 'web',
        ])->first();

        /** @var User $subscriber */
        foreach ($subscribers as $subscriber) {
            $this->when = $this->getInterval($type, $proposable);
            $this->route = $subscriber;

            if (! $this->checkConditions($type, $proposal, $subscriber) || ! $this->checkUniqueness($type, $proposal, $subscriber)) {
                return;
            }

            $this->when = $this->getInterval($type, $proposable);
            $this->route = $subscriber;

            $md5Data = [
                'uuid' => $proposable->uuid,
                'subscriber' => $subscriber->uuid,
                'steps' => $steps,
                'cycles' => $proposal->cycles->count(),
                'review' => true,
            ];

            $md5 = md5((string) json_encode($md5Data));

            $data = [
                'to' => $subscriber->uuid,
                'url' => $notificationAction === 'decline' ? 'edit/'.class_to_type($proposal->proposable_type).'/'.$proposal->proposable_uuid : null,
                'steps' => $steps,
                'sendMail' => $proposal->status === 1 ? $proposal->workflow->send_mail : $workflowCurrentStep->send_mail,
                'md5' => $md5,
            ];

            if (Notification::query()->where('data', 'LIKE', '%"md5":"'.$md5.'"%')->exists()) {
                continue;
            }

            $customNotifications = config('workflow.custom_notifications', false);

            if ($customNotifications && is_callable($customNotifications)) {
                $customNotifications($data, $proposable, 'web', 'accept');
            }

            if (! config('workflow.delay_notifications', true)) {
                $this->when = Carbon::now();
            }

            $this->notify($type, $subscriber, $proposable, $data);

            if (! config('workflow.delay_notifications')) {
                $this->when = null;
            }
        }

        // If the content is accepted, we send publication notifications to subscribed users / roles
        if ($proposal->status === 1) {
            $via = ['web'];
            if ($proposal->workflow->send_mail) {
                $via[] = 'email';
            }

            $types = NotificationType::query()
                ->where('model', get_class($proposable))
                ->whereIn('action', ['create', 'createOrUpdate'])
                ->whereIn('via', $via)
                ->get();

            foreach ($types as $type) {
                if ($type->role) { // role notifications
                    if ($type->roles->isEmpty()) {
                        continue;
                    }

                    $this->when = $this->getInterval($type, $proposable);
                    $this->route = $type->roles->first(); // We create only one notification (to avoid double notifications)
                    $this->notify($type, null, $proposable, ['from' => $author->uuid], $type->roles->pluck('id'));
                    continue;
                }

                $type->subscribers->each(
                    function (User $subscriber) use ($type, $proposable, $author) {
                        $this->when = $this->getInterval($type, $proposable);
                        $this->route = $subscriber;

                        if (! $this->checkConditions($type, $proposable, $subscriber) || ! $this->checkUniqueness($type, $proposable, $subscriber)) {
                            return;
                        }

                        $this->notify($type, $subscriber, $proposable, ['from' => $author->uuid]);
                    }
                );
            }
        }
    }
}
