<?php

declare(strict_types=1);

namespace Inside\Workflow\Listeners;

use Illuminate\Support\Facades\Cache;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Content\Models\Content;
use Inside\Content\Models\Section;
use Inside\Permission\Models\User;
use Inside\Workflow\Events\ProposalEditedEvent;
use Inside\Workflow\Facades\Proposal;
use Inside\Workflow\Facades\Workflow;

final class ContentUpdatedListener
{
    public function handle(ContentFullyUpdatedEvent $event): void
    {
        if (! Workflow::isWorkflowEnable()) {
            return;
        }

        $content = $event->model;
        if ($content instanceof Section || is_null($content->uuid)) {
            return;
        }

        $proposal = Proposal::getFromContent($content->uuid, get_class($content));

        if (Proposal::bypassWorkflow($content)) {
            return;
        }

        // For content declined we attach the workflow proposal at the edition
        if (is_null($proposal)) {
            $author = Proposal::getContentAuthorUuid($content);
            $proposal = Proposal::attachToContent($content, $author);

            // If workflow attached we depublish the content
            // the publication will be done by the workflow
            // We also trigger the proposal edited event
            if ($proposal) {
                $languages = list_languages();

                if (count($languages) > 1 && config('workflow.multilingual_type') === 'automatic') {
                    foreach ($languages as $language) {
                        if ($content->langcode === $language) {
                            continue;
                        }

                        $translation = $content->getTranslationIfExists($language);

                        if ($translation->langcode === $content->langcode) {
                            continue;
                        }

                        Proposal::attachToContent($translation, $author);
                    }
                }

                ProposalEditedEvent::dispatch($proposal, $content);

                if ($content->status !== 0) {
                    Proposal::editContentStatus($content, false);
                }
            }

            return;
        }

        if (Proposal::isContributor($proposal, User::find(Proposal::getContentAuthorUuid($content)))) {
            // If the current user is contributor and the proposal has been rejected or accepted
            // We add a new cycle
            // We send the edit event for notify reviewer
            $nextStep = Proposal::getNextStep($proposal->id);

            if (! $nextStep || ! Proposal::isFirstStep($nextStep->id) || $proposal->cycles->isEmpty()) {
                Proposal::addCycle($proposal->id);
                ProposalEditedEvent::dispatch($proposal, $content);
            }
        }
    }
}
