<?php

namespace Inside\Wall\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Database\Eloquent\Builder;
use Inside\Notify\Models\NotificationType;
use Laravel\Lumen\Routing\Controller;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Inside\Notify\Models\NotificationSubscriber;
use Illuminate\Http\Request;
use Inside\Content\Models\Contents\Comments;
use Inside\Content\Transformers\ContentTransformer;

class WallsController extends Controller
{
    public function subscribe(string $wallUuid): JsonResponse
    {
        /** @var User|null $user */
        $user = Auth::user();
        if (!$user) {
            throw new AccessDeniedHttpException('forbidden');
        }

        /** @var NotificationType $notificationType */
        $notificationType = NotificationType::query()
            ->where('condition', 'walls:' . $wallUuid)
            ->where('action', 'contentCommented')
            ->first();

        NotificationSubscriber::firstOrCreate(
            [
                'user_uuid' => $user->uuid,
                'notification_type_id' => $notificationType->id,
            ]
        );

        return new JsonResponse(['subscribed' => true]);
    }

    public function unsubscribe(string $wallUuid): JsonResponse
    {
        /** @var User|null $user */
        $user = Auth::user();
        if (!$user) {
            throw new AccessDeniedHttpException('forbidden');
        }

        /** @var NotificationType $notificationType */
        $notificationType = NotificationType::query()
            ->where('condition', 'walls:' . $wallUuid)
            ->where('action', 'contentCommented')
            ->first();

        NotificationSubscriber::where([
            'user_uuid' => $user->uuid,
            'notification_type_id' => $notificationType->id,
        ])->delete();

        return new JsonResponse(['unsubscribed' => true]);
    }

    public function isSubscribed(string $wallUuid): JsonResponse
    {
        /** @var User|null $user */
        $user = Auth::user();
        if (!$user) {
            throw new AccessDeniedHttpException('forbidden');
        }

        /** @var NotificationType $notificationType */
        $notificationType = NotificationType::query()
            ->where('condition', 'walls:' . $wallUuid)
            ->where('action', 'contentCommented')
            ->first();

        return new JsonResponse(
            [
                'isSubscribed' => NotificationSubscriber::where([
                    'user_uuid' => $user->uuid,
                    'notification_type_id' => $notificationType->id,
                ])->exists(),
            ]
        );
    }

    public function getAllComments(Request $request, ContentTransformer $transformer): JsonResponse
    {
        /** @var User|null $user */
        $user = Auth::user();

        /** @var string $filters */
        $filters = $request->query('filters') ?? '';
        $filters = json_decode($filters, true);
        $langcode = $filters['langcode'] ?? $user->langcode;

        $query = Comments::with('walls')
            ->where('langcode', $langcode)
            ->whereNull('pid')
            ->when(
                !blank($filters['status:eq']),
                fn (Builder $query) => $query->where('status', $filters['status:eq'])
            )->whereHas('walls', function ($query) use ($filters) {
                if (isset($filters['walls'])) {
                    collect($filters['walls'])
                        ->each(fn ($value, $name) => $query->where($name, $value));
                }
            })->orderBy('created_at', 'DESC')
            ->when(
                !blank($filters['limit']),
                fn (Builder $query) => $query->limit($filters['limit'])
            );


        $comments = $transformer->transformCollection(
            collection: $query->get(),
            fields: ['uuid', 'body', 'file', 'tree', 'reactions' => ['count', 'users' => ["firstname", "lastname", "image", "uuid"]], 'created_at', 'authors' => ['firstname', 'lastname', 'email', 'image'], 'walls' => ['title', 'preview_image', 'slug']],
        );

        return response()->json([
            'data' => $comments['data'] ?? [],
        ]);
    }
}
