<?php

declare(strict_types=1);

namespace Inside\Vacation\Services;

use Illuminate\Support\Collection;
use Inside\Content\Models\Contents\Vacations;
use Inside\Content\Models\Contents\VacationsStatus;
use Inside\Host\Bridge\BridgeContent;
use Inside\Vacation\Contracts\VacationValidator;
use Inside\Vacation\Models\VacationValidation;

final class VacationValidatorService implements VacationValidator
{
    public function computeVacationStatus(Vacations $vacation): void
    {
        /** @var Collection $validators */
        $validators = $vacation->validator->pluck('uuid');

        VacationValidation::query()
            ->whereNotIn('user_uuid', $validators)
            ->where('model_uuid', $vacation->uuid)
            ->delete();

        $validations = VacationValidation::query()
            ->whereIn('user_uuid', $validators)
            ->where('model_uuid', $vacation->uuid)
            ->get();

        $status = match (true) {
            $validations->contains('status', self::STATUS_REFUSED) => self::STATUS_REFUSED,
            $validations->count() === $validators->count() && $validations->every('status', self::STATUS_VALIDATED) => self::STATUS_VALIDATED,
            default => self::STATUS_IN_PROGRESS
        };

        $this->updateValidationStatus($vacation, $status);
    }

    private function updateValidationStatus(Vacations $vacation, string $statusValue): void
    {
        $status = VacationsStatus::where('value', $statusValue)->where('langcode', $vacation->langcode)->first();

        if (! $status instanceof VacationsStatus) {
            throw new \LogicException("The vacation status $status does not exist for the langcode $vacation->langcode");
        }

        if ($vacation->vacationsStatus->contains($status)) {
            return;
        }

        (new BridgeContent())->contentUpdate($vacation->content_type, ['uuid' => $vacation->uuid, 'vacations_status' => $status->uuid]);
    }
}
