<?php

namespace Inside\Vacation\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentSynchedEvent;
use Inside\Content\Models\Contents\Vacations;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

final class VacationsUpdatedListener extends BaseNotificationListener implements ShouldQueue
{
    /**
     * @param ContentSynchedEvent $event
     * @return void
     */
    public function handle(ContentSynchedEvent $event): void
    {
        $model = $event->model;
        $fieldName = $event->fieldName;
        /** @var User $user */
        $user = $event->user;

        if (! $model instanceof Vacations || ! $user instanceof User) {
            return;
        }

        if ($fieldName === 'vacations_status') {
            $this->statusUpdated($model, $user);
        }

        if ($fieldName === 'validator') {
            $this->validatorUpdated($model, $user);
        }
    }

    /**
     * @param Vacations $model
     * @param User $user
     * @return void
     */
    private function validatorUpdated(Vacations $model, User $user): void
    {
        $notificationTypes = NotificationType::where('model', type_to_class($model->content_type))
            ->where('action', 'create')
            ->get();

        User::query()
            ->whereIn('uuid', $model->validator->pluck('uuid'))
            ->each(function (User $targetUser) use ($model, $notificationTypes, $user) {
                if ($targetUser->uuid === $user->uuid) {
                    return;
                }

                $this->route = $targetUser;
                $notificationTypes->each(function (NotificationType $notificationType) use ($model, $targetUser) {
                    $this->when = $this->getInterval($notificationType);
                    $this->notify(
                        $notificationType,
                        $targetUser,
                        $model,
                        [
                            'from' => $model->author,
                            'url' => 'conges/gerer',
                        ]
                    );
                });
            });
    }

    /**
     * @param Vacations $model
     * @param User $user
     * @return void
     */
    private function statusUpdated(Vacations $model, User $user): void
    {
        $targetUser = $model->creator;

        if (! $targetUser instanceof User || $targetUser->uuid === $user->uuid) {
            return;
        }

        $status = $model->vacationsStatus->first();

        $start = get_date($model->date)?->format('d-m-Y H:i');
        $end = get_date($model->end_date)?->format('d-m-Y H:i');
        $from = $model->validator()->first()->uuid;

        $this->route = $targetUser;

        NotificationType::where('model', type_to_class($model->content_type))
            ->where('action', 'statusUpdated')
            ->each(function (NotificationType $notificationType) use ($status, $targetUser, $model, $from, $start, $end) {
                $this->when = $this->getInterval($notificationType);
                $this->notify(
                    $notificationType,
                    $targetUser,
                    $model,
                    [
                        'from' => $from,
                        'vacations_status' => $status->title,
                        'date_start' => $start,
                        'date_end' => $end,
                        'url' => 'conges/consulter',
                        'text' => "notifications.create.content.vacations.status.$status->value.text",
                        'extra' => [
                            'mail' => [
                                'subject' => "notifications.create.content.vacations.status.$status->value.mail.subject",
                                'text' => "notifications.create.content.vacations.status.$status->value.mail.subject",
                            ]
                        ]
                    ]
                );
            });
    }
}
