<?php

declare(strict_types=1);

namespace Inside\Vacation\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\VacationsStatus;
use Inside\Vacation\Contracts\VacationValidator;
use Inside\Vacation\Models\VacationValidation;
use Symfony\Component\HttpFoundation\Response;

final class InjectValidationStatusesMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $response = $next($request);

        if ($response instanceof Response && $response->isOk() && $this->isVacationsRequest($request)) {
            return $this->injectValidationStatuses($response);
        }

        return $response;
    }

    public function isVacationsRequest(Request $request): bool
    {
        $path = $request->path();
        $method = $request->getMethod();

        return $method === 'GET' && $path === "api/v1/content/vacations";
    }

    public function injectValidationStatuses(Response $response): Response
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            return $response;
        }

        $data = json_decode_response($response);

        $data['data'] = collect($data['data'])
            ->map(function (array $vacation) use ($user) {
                $validators = collect($vacation['validator']['data'] ?? []);

                if ($validators->isEmpty() || ! isset($vacation['uuid'])) {
                    return $vacation;
                }

                $vacation['validator']['data'] = $validators->map(function ($validator) use ($user, $vacation) {
                    if (! isset($validator['uuid'])) {
                        return $validator;
                    }

                    /** @var ?VacationValidation $validation */
                    $validation = VacationValidation::query()
                        ->where('model_uuid', $vacation['uuid'])
                        ->where('user_uuid', $validator['uuid'])
                        ->first();

                    $status = VacationsStatus::query()
                        ->where('value', $validation?->status ?? VacationValidator::STATUS_IN_PROGRESS)
                        ->where('langcode', $user->langcode)
                        ->first();

                    if ($status instanceof VacationsStatus) {
                        $validator['vacations_status'] = [
                            'uuid' => $status->uuid,
                            'value' => $status->value,
                            'title' => $status->title,
                            'langcode' => $status->langcode,
                        ];
                    }

                    return $validator;
                });

                return $vacation;
            })
            ->all();

        set_response($response, $data);

        return $response;
    }
}
