<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Collection;
use Inside\Content\Models\Contents\VacationsStatus;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Migrations\CreateContentTypes;
use Inside\Vacation\Contracts\VacationValidator;

final class CreateVacationsStatusContents extends Migration
{
    use CreateContentTypes;

    private ?BridgeContent $bridgeContent = null;

    public function getBridgeContent(): BridgeContent
    {
        return $this->bridgeContent ??= new BridgeContent();
    }

    protected function createContent(array $data, string $langcode, ?VacationsStatus $original = null): ?string
    {
        $data = collect($data)->map(fn (mixed $value) => $value[$langcode] ?? $value)
            ->put('langcode', $langcode)
            ->when($original !== null, fn (Collection $data) => $data->put('uuid_host', $original->uuid_host))
            ->all();

        $existing = VacationsStatus::query()
            ->where('langcode', $langcode)
            ->where('title', $data['title'])
            ->first();

        if ($existing instanceof VacationsStatus) {
            $data['uuid'] = $existing->uuid;
        }

        return $this->getBridgeContent()->contentInsert(
            type: 'vacations_status',
            data: $data,
            creation: $existing === null,
            fromCli: true
        );
    }

    protected function createVacationsStatuses(): void
    {
        $locale = config('app.locale');
        $languages = collect(list_languages())->reject($locale);

        $statuses = collect([
            [
                'value' => VacationValidator::STATUS_VALIDATED,
                'title' => [
                    'fr' => 'Validée',
                    'en' => 'Validated',
                    'es' => 'Validado',
                    'de' => 'Freigegeben',
                    'pt' => 'Validada',
                    'nl' => 'Gevalideerd',
                ],
            ],
            [
                'value' => VacationValidator::STATUS_REFUSED,
                'title' => [
                    'fr' => 'Refusée',
                    'en' => 'Refused',
                    'es' => 'Rechazado',
                    'de' => 'Abgelehnt',
                    'pt' => 'Recusada',
                    'nl' => 'Geweigerd',
                ],
            ],
            [
                'value' => VacationValidator::STATUS_IN_PROGRESS,
                'title' => [
                    'fr' => 'En cours',
                    'en' => 'In progress',
                    'es' => 'En curso',
                    'de' => 'In Arbeit',
                    'pt' => 'Em andamento',
                    'nl' => 'In behandeling',
                ],
            ],
        ]);

        $statuses->each(function (array $data) use ($locale, $languages) {
            $original = $this->createContent($data, $locale);
            $original = VacationsStatus::find($original);

            if ($original instanceof VacationsStatus) {
                $languages->each(fn (string $language) => $this->createContent($data, $language, $original));
            }
        });
    }

    public function up(): void
    {
        $this->bootUp(__FILE__);
        $this->migrateContent('vacations_status');
        $this->createVacationsStatuses();
        $this->finishUp();
    }
}
