<?php

namespace Inside\OneLogin\Importers;

use Illuminate\Support\Arr;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use Inside\OneLogin\Services\OneLoginApiService;
use OneLogin\api\models\User;

class OneLoginImporter implements ImporterInterface
{
    public function prepareImport(): bool
    {
        return true;
    }

    public function getSyncAttributes(mixed $user, array $entry, BridgeContent $bridge): array
    {
        $syncAttributes = $entry['sync_attributes'] ?? ['email' => 'email', 'name' => 'username'];

        $discoveringAttribute = $entry['discovering_attribute'] ?? 'username';

        if (! isset($syncAttributes[$discoveringAttribute])) {
            $syncAttributes[$discoveringAttribute] = $discoveringAttribute;
        }

        $customAttributes = [];
        $oneLoginService = OneLoginApiService::load($entry);
        $customUser = $oneLoginService->getClient()->getUser(Arr::get((array)$user, 'id'));

        if (!empty($entry['custom_attributes'])) {
            foreach ($entry['custom_attributes'] as $customAttrKey => $customAttrValue) {
                $customAttributes[$customAttrKey] = Arr::get($customUser->getUserCustomAttributes(), $customAttrValue);
            }
        }

        if (!empty($entry['manager_attributes']) && !empty(Arr::get((array)$customUser, 'managerUserId'))) {
            $oneLoginService = OneLoginApiService::load($entry);
            $managerUser = $oneLoginService->getClient()->getUser(Arr::get((array)$customUser, 'managerUserId'));
            foreach ($entry['manager_attributes'] as $customAttrKey => $customAttrValue) {
                $customAttributes[$customAttrKey] = Arr::get($managerUser->getUserParams() ?? [], $customAttrValue); // no-sonar
            }
        }

        collect($customAttributes)->each(function ($value, $key) use (&$user) {
            if (is_array($user)) {
                $user[$key] = $value;
            } else {
                $user->$key = $value;
            }
        });

        $data = collect($syncAttributes)->map(function (mixed $attribute) use ($user, $bridge) {
            if (! isset($attribute)) {
                return null;
            }
            return is_callable($attribute) ? $attribute((array) $user, $bridge) : Arr::get((array) $user, $attribute);
        })->all();

        $postSync = config('onelogin.post_sync');

        if (is_callable($postSync)) {
            $postSync($data);
        }

        return $data;
    }

    public function getUsers(array $entry, ?string $filter, ?string $test = null, int $limit = 0): array
    {
        $users = collect(OneLoginApiService::load($entry)->getClient()->getUsers(['email' => $filter]));
        $postFilter = $entry['postfilter'];

        if (is_callable($postFilter)) {
            $users = $postFilter($users);
        }

        return $users->map(fn (User $user) => array_merge($user->getUserParams(), ['id' => $user->id]))->all();
    }

    public function getEntries(): array
    {
        $default = config('onelogin');

        $entries = $default['onelogin'] ?? [];

        if (empty($entries)) {
            return ['default' => $default];
        }

        return array_map(fn (array $entry) => array_merge($default, $entry), $entries);
    }

    public function getUserInformation(?string $username, array $entry): ?array
    {
        /** @var ?User $user */
        $user = array_first(OneLoginApiService::load($entry)->getClient()->getUsers(['email' => $username], 1));

        return array_merge($user?->getUserParams(), ['id' => $user?->id]);
    }

    public function getCleanData()
    {
        return config('onelogin.cleandata');
    }

    public function getPreFlight()
    {
        return config('onelogin.preflight');
    }

    public function getPostFlight()
    {
        return config('onelogin.postflight');
    }

    public function getFilter(): ?string
    {
        return config('onelogin.filter');
    }
}
