<?php

namespace Inside\Okta\Services;

use Firebase\JWT\JWT;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Log;
use Psr\Http\Message\ResponseInterface;

class OktaApiService
{
    private ?string $privateKey = null;

    private ?string $accessToken = null;

    private ?Client $client = null;

    private function __construct(
        private string $url,
        private string $clientId,
        private ?string $scope,
        private ?string $privateKeyPath = null,
    ) {
    }

    public static function load(array $config): self
    {
        return new self(
            url: $config['url'],
            clientId: $config['client_id'],
            scope: $config['scope'] ?? null,
            privateKeyPath: $config['private_key_path'] ?? null,
        );
    }

    private function getPrivateKey(): string
    {
        if ($this->privateKey) {
            return $this->privateKey;
        }

        if (! $this->privateKeyPath) {
            throw new \Exception("Okta: missing private key path");
        }

        $privateKey = file_get_contents($this->privateKeyPath);

        if (! is_string($privateKey)) {
            throw new \Exception("Okta: Invalid private key path ($this->privateKeyPath)");
        }

        return $this->privateKey = $privateKey;
    }

    public function getClient(): Client
    {
        return $this->client ??= new Client(['base_uri' => $this->url]);
    }

    private function getResponseContent(ResponseInterface $response): array
    {
        return json_decode($response->getBody()->getContents(), true);
    }

    public function createClientAssertionToken(): string
    {
        $payload = [
            'aud' => "$this->url/oauth2/v1/token",
            'iss' => $this->clientId,
            'sub' => $this->clientId,
            'exp' => (string) now()->addHour()->timestamp,
        ];

        return JWT::encode($payload, $this->getPrivateKey(), 'RS256');
    }

    public function createAccessToken(): string
    {
        $response = $this->getClient()->post('/oauth2/v1/token', [
            'form_params' => [
                'client_assertion' => $this->createClientAssertionToken(),
                'client_assertion_type' => 'urn:ietf:params:oauth:client-assertion-type:jwt-bearer',
                'grant_type' => 'client_credentials',
                'scope' => $this->scope,
            ],
        ]);

        $content = $this->getResponseContent($response);

        return $content['access_token'];
    }

    public function getAccessToken(): string
    {
        return $this->accessToken ??= $this->createAccessToken();
    }

    private function getNextPageUrl(ResponseInterface $response): ?string
    {
        return collect($response->getHeader('link'))
            ->filter(fn (string $header) => str($header)->contains('rel="next"'))
            ->map(fn (string $header) => str($header)->between('<', '>')->toString())
            ->first();
    }

    public function listAppUsers(?string $username = null, int $limit = 0): array
    {
        try {
            $response = $this->getClient()->get("/api/v1/apps/$this->clientId/users", [
                'headers' => ['Authorization' => "Bearer {$this->getAccessToken()}"],
                'query' => ['q' => $username],
            ]);

            $users = collect($this->getResponseContent($response));

            while (($next = $this->getNextPageUrl($response))) {
                $response = $this->getClient()->get($next, ['headers' => ['Authorization' => "Bearer {$this->getAccessToken()}"]]);
                $users = $users->concat($this->getResponseContent($response));
            }

            return $users->values()->all();
        } catch (ClientException $exception) {
            Log::error("[OktaApiService::listAppUsers] Failure: {$exception->getMessage()}");

            $response = $exception->getResponse();

            if ($response instanceof ResponseInterface) {
                Log::error("[OktaApiService::listAppUsers] Failure (Http Response)", $this->getResponseContent($response));
            }

            return [];
        }
    }
}
