<?php

namespace Inside\Okta\Importers;

use Illuminate\Support\Arr;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use Inside\Okta\Services\OktaApiService;

class OktaImporter implements ImporterInterface
{
    public function prepareImport(): bool
    {
        return true;
    }

    public function getSyncAttributes(mixed $user, array $entry, BridgeContent $bridge): array
    {
        $syncAttributes = $entry['sync_attributes'] ?? ['email' => 'mail', 'name' => 'name'];

        $discoveringAttribute = $entry['discovering_attribute'] ?? 'email';

        if (! isset($syncAttributes[$discoveringAttribute])) {
            $syncAttributes[$discoveringAttribute] = $discoveringAttribute;
        }

        $data = collect($syncAttributes)
            ->map(function (mixed $oktaValue) use ($user, $bridge) {
                if (! isset($oktaValue)) {
                    return null;
                }

                return is_callable($oktaValue) ? $oktaValue($user, $bridge) : Arr::get($user, $oktaValue);
            })
            ->all();

        $postSync = config('okta.post_sync');

        if (is_callable($postSync)) {
            $postSync($data);
        }

        return $data;
    }

    public function getUsers(array $entry, ?string $filter, ?string $test = null, int $limit = 0): array
    {
        $users = collect(OktaApiService::load($entry)->listAppUsers(username: null, limit: $limit));

        $postFilter = $entry['postfilter'];

        if (is_callable($postFilter)) {
            $users = $postFilter($users);
        }

        return $users->all();
    }

    public function getEntries(): array
    {
        $default = config('okta');

        $entries = $default['entries'] ?? [];

        if (empty($entries)) {
            return ['default' => $default];
        }

        return array_map(fn (array $entry) => array_merge($default, $entry), $entries);
    }

    public function getUserInformation(?string $username, array $entry): ?array
    {
        return Arr::first(OktaApiService::load($entry)->listAppUsers(username: $username));
    }

    public function getCleanData()
    {
        return config('okta.cleandata');
    }

    public function getPreFlight()
    {
        return config('okta.preflight');
    }

    public function getPostFlight()
    {
        return config('okta.postflight');
    }

    public function getFilter(): ?string
    {
        return config('okta.filter');
    }
}
