<?php

namespace Inside\LDAP\LDAP;

use Adldap\Laravel\Facades\Adldap;
use Closure;
use Illuminate\Support\Arr;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;

class LDAPImporter implements ImporterInterface
{
    /**
     * Function supposed to initialize everything for the import
     *
     * @return bool
     */
    public function prepareImport(): bool
    {
        chdir(dirname(base_path(), 3));

        return true;
    }

    /**
     * Return user data after synchronized attribute with the type of import (syn_attributes in the config file)
     *
     * @param mixed $user
     * @param array $entry
     * @param BridgeContent $bridge
     * @return array
     */
    public function getSyncAttributes($user, $entry, $bridge): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        $syncAttributes = $entry['sync_attributes'] ?? [
                'email' => 'userprincipalname',
                'name'  => 'cn',
            ];
        // Note: charles indique d'utiliser le cn comme référence ldap
        $discoveringAttribute = $entry['discovering_attribute'] ?? 'cn';
        if (!isset($syncAttributes[$discoveringAttribute])) {
            $syncAttributes[$discoveringAttribute] = $discoveringAttribute;
        }
        $data = [];
        foreach ($syncAttributes as $modelField => $ldapField) {
            if (($ldapField instanceof Closure)) {
                $data[$modelField] = $ldapField($user, $bridge);
            } else {
                $data[$modelField] = $user->getFirstAttribute($ldapField);
            }
        }

        return $data;
    }

    /**
     * Get all users from the base provider (google, csv file, etc.)
     *
     * @param array $entry
     * @param array|null $filter
     * @param array|null $test
     * @return array
     */
    public function getUsers($entry, $filter, $test = null, int $limit = 0): array
    {
        // Create user from test file
        if ($test) {
            $test = Adldap::getDefaultProvider()->make()->user($test);

            return [$test];
        }

        if (!$entry['auto_connect']) {
            Adldap::getProvider($entry['connection'])->connect();
        }

        // TODO gérer les users en fonction de $entry (Adldap::getProvider($entry['providerName']) ?)
        $query = Adldap::getProvider($entry['connection'])->search()->users();
        if ($filter) {
            $query->rawFilter($filter);
        }

        return $query->paginate()->getResults();
    }

    /**
     * Get all entries from the base provider
     * (example: for google, one entry is a domain)
     * (example: for csv, one entry is one file)
     *
     * if there is no several entry, must return ['default' => entry] (something like that)
     *
     * @return array
     */
    public function getEntries(): array
    {
        $connections = config('ldap.connections');

        $entries = [];

        foreach (array_keys($connections) as $connection) {
            $entries[$connection]                 = config('ldap_auth');
            $entries[$connection]['connection']   = $connection;
            $entries[$connection]['auto_connect'] = $connections[$connection]['auto_connect'];

            if (array_key_exists('sync_attributes', $connections[$connection])) {
                $entries[$connection]['sync_attributes'] = $connections[$connection]['sync_attributes'];
            }

            if (array_key_exists('discovering_attribute', $connections[$connection])) {
                $entries[$connection]['discovering_attribute'] = $connections[$connection]['discovering_attribute'];
            }
        }

        return $entries;
    }

    /**
     * Get the function cleanData (launched after retrieving data for a given user) from config file
     * Signature must be : function(&$data)
     */
    public function getCleanData(): mixed
    {
        return config('ldap_auth.cleandata');
    }

    /**
     * Get the function preFlight (launched before the import) from the config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     */
    public function getPreFlight(): mixed
    {
        return config('ldap_auth.preflight', null);
    }

    /**
     * Get the function postFlight (launched after the import) from config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     */
    public function getPostFlight(): mixed
    {
        return config('ldap_auth.postflight', null);
    }

    /**
     * Get filters (from .env or config files, depends of import type)
     */
    public function getFilter(): ?string
    {
        return config('ldap_auth.filter', null) ?? env('ADLDAP_FILTER', null);
    }

    /**
     * Get User information from AD, null if user not found.
     *
     * @param string|null $username
     * @param array $entry
     * @return array|null
     */
    public function getUserInformation($username, $entry): ?array
    {
        if ($username) {
            if (!$entry['auto_connect']) {
                Adldap::getProvider($entry['connection'])->connect();
            }

            $attribute = config('ldap_auth.discovering_attribute') ?? 'cn';

            $user = Adldap::getProvider($entry['connection'])->search()->where($attribute, '=', $username)->get()->first();

            return $user ? $user->getAttributes() : [];
        }

        $users = $this->getUsers($entry, null);

        return Arr::first($users)->getAttributes();
    }
}
