<?php

namespace Inside\Google\Google;

use Closure;
use Exception;
use Google\Exception as GoogleException;
use Google_Client;
use Google_Service_Directory;
use Google_Service_Directory_User;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use Inside\Support\Str;

class GoogleImporter implements ImporterInterface
{
    protected Google_Client $client;

    protected Google_Service_Directory $service;

    /**
     * @param Collection|Google_Service_Directory_User[] $users
     * @param string $fieldName
     * @param array $values
     * @return array
     */
    protected function excludeUsers(Collection $users, string $fieldName, array $values): array
    {
        $filteredUsers = $users->reject(function ($user) use ($fieldName, $values) {
            if ($fieldName === 'group') {
                $userGroupList = $this->service->groups->listGroups([
                    'userKey' => $user->getPrimaryEmail(),
                ]);

                $diff = collect($values)->diff(collect($userGroupList)->pluck('email'));
                return $diff->isNotEmpty();
            }

            $field = Arr::get((array)$user, $fieldName);
            return in_array($field, $values);
        });
        return $filteredUsers->toArray();
    }

    public function getEntries(): array
    {
        $defaultConfig = config('google');
        $domainsConfigs = $defaultConfig['google_domains'] ?? [];
        unset($defaultConfig['google_domains']);
        if (empty($domainsConfigs)) {
            return ['default' => $defaultConfig];
        }

        $entries = [];
        foreach ($domainsConfigs as $domain => $configs) {
            $entries[$domain] = array_merge($defaultConfig, $configs);
        }

        return $entries;
    }

    public function prepareImport(): bool
    {
        $this->client = new Google_Client();
        $this->client->setAccessType('offline');
        $this->client->setUseBatch(false);

        return true;
    }

    /**
     * @param Google_Service_Directory_User $user
     * @param array $entry
     * @param BridgeContent $bridge
     * @return array
     */
    public function getSyncAttributes($user, array $entry, BridgeContent $bridge): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        $syncAttributes = $entry['sync_attributes'] ?: [
            'email' => 'getPrimaryEmail',
            'name' => 'getPrimaryEmail',
        ];

        $discoveringAttribute = $entry['discovering_attribute'] ?? 'getPrimaryEmail';
        if (!isset($syncAttributes[$discoveringAttribute])) {
            $syncAttributes[$discoveringAttribute] = $discoveringAttribute;
        }


        $data = [];
        foreach ($syncAttributes as $modelField => $googleField) {
            if (($googleField instanceof Closure)) {
                $data[$modelField] = $googleField($user, $bridge, $this->service);
            } else {
                if (!method_exists($user, $googleField)) {
                    continue;
                }

                $data[$modelField] = $user->{$googleField}();
            }
        }
        return $data;
    }

    /**
     * @param array $entry
     * @param string|null $filter
     * @param array|null $test
     * @return array|Google_Service_Directory_User[]
     * @throws GoogleException
     */
    public function getUsers($entry, $filter, $test = null, int $limit = 0): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        if ($test) {
            $user = new Google_Service_Directory_User($test);
            return [$user];
        }

        if (isset($entry['service_account_key_b64'])) {
            $json = base64_decode($entry['service_account_key_b64'], true);

            if ($json === false) {
                throw new Exception('Invalid base64 encoding for service account key.');
            }

            $config = json_decode($json, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid JSON in service account key.');
            }

            $this->client->setAuthConfig($config);
        } else {
            $this->client->setAuthConfig($entry['service_account_key_path']);
        }

        $this->client->setClientId($entry['client_id']);
        $this->client->setClientSecret($entry['client_secret']);
        $this->client->setSubject($entry['service_account_name']);
        $this->client->setHostedDomain($entry['hd']);

        $this->client->setScopes($entry['scopes']);
        $this->service = new Google_Service_Directory($this->client);


        $orgUnitPaths = data_get($entry, 'org_unit_paths', '/');
        $orgUnitPaths = is_array($orgUnitPaths) ? $orgUnitPaths : [$orgUnitPaths];
        $users = [];
        foreach ($orgUnitPaths as $orgUnitPath) {
            $nextPageToken = false;
            do {
                $optParams = [
                    'customer' => 'my_customer',
                    'pageToken' => $nextPageToken,
                    'maxResults' => 200,
                    'query' => sprintf("orgUnitPath='%s'", addcslashes($orgUnitPath, "'\\")),
                    'projection' => 'full',
                    'showDeleted' => 'false',
                ];
                $list = $this->service->users->listUsers($optParams);
                $users = array_merge($users, $list->getUsers());
                $nextPageToken = $list->getNextPageToken();
            } while ($nextPageToken);
        }

        // you have to give the path of the variables you want to exclude, even if it's in array
        // the path has to be separated by dots, the values to exclude by ',' and an = between them
        // exemple: php artisan user:import google --filter=exclude:organizations.0.description=Freelance,"compte de service"
        // separate it with ; character if you want to exclude multiple fields
        // exemple: php artisan user:import google --filter="exclude:suspended=true;websites.0.value=https://www.non-intranet.fr" --disable-not-imported

        if ($filter !== null && Str::startsWith($filter, 'exclude:')) {
            $filter = Str::after($filter, 'exclude:');
            $filters = explode(";", $filter);
            foreach ($filters as $toFilter) {
                $array = preg_split("/[=,]+/", $toFilter);
                if ($array !== false) {
                    $users = $this->excludeUsers(collect($users), $array[0], array_slice($array, 1));
                }
            }
        }

        foreach ($users as $user) {
            $thumbnailPhotoUrl = $user->getThumbnailPhotoUrl();
            if ($thumbnailPhotoUrl) {
                $user->thumbnailPhotoUrl = preg_replace('/=.+$/', '', $thumbnailPhotoUrl);
            }
        }

        return $users;
    }

    /**
     * Get the function cleanData (launched after retrieving data for a given user) from config file
     * Signature must be : function(&$data)
     */
    public function getCleanData(): mixed
    {
        return config('google.cleandata');
    }

    public function getPreFlight(): mixed
    {
        return config('google.preFlight', null);
    }

    public function getPostFlight(): mixed
    {
        return config('google.postFlight', null);
    }

    public function getFilter(): ?string
    {
        return config('google.filter', null);
    }

    /**
     * Get User information from AD, null if user not found.
     *
     * @param string|null $username
     * @param array $entry
     * @return array|null
     * @throws Exception
     */
    public function getUserInformation($username, $entry): ?array
    {
        $users = $this->getUsers($entry, null);

        if ($username === null) {
            return (array)Arr::first($users);
        }

        foreach ($users as $user) {
            if ($user->getPrimaryEmail() === $username) {
                return (array)$user;
            }
        }
        return null;
    }
}
