<?php

namespace Inside\CSV\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Inside\CSV\Actions\Export\CsvExportStrategy;
use Inside\CSV\Actions\Export\ExcelExportStrategy;
use Inside\CSV\Actions\Export\ExportStrategy;
use Inside\CSV\Actions\FeedValidation\CsvFeedValidationStrategy;
use Inside\CSV\Actions\FeedValidation\FeedValidationStrategy;
use Inside\CSV\Actions\FeedValidation\XlsxFeedValidationStrategy;
use Inside\CSV\Exceptions\FileNotFoundException;
use Inside\CSV\Exceptions\HeadersMissingException;
use Inside\CSV\Exceptions\InvalidExtensionException;
use Inside\CSV\Traits\FeedConfig;

class FeedValidationService
{
    use FeedConfig;

    private const STEP_WEIGHT = 20;
    private string $file;
    private string $validationToken;
    private int $progress = 0;

    public function __construct()
    {
    }

    public function execute(string $file, string $validationToken): void
    {
        try {
            $extension = $this->getExtension($file);
            $validator = $this->getValidator($extension);
            $expectedColumns = $this->getExpectedColumns($extension);


            $this->file = Storage::disk('local')->path($file);
            $this->validationToken = $validationToken;
            $validator->validate(
                $this->file,
                $expectedColumns,
                fn ($errors) => $this->handleValidationStep($errors)
            );
        } catch (\Exception $exception) {
            $this->addErrors([$exception->getMessage()]);
        } finally {
            Cache::tags([$validationToken])->put('state', 100, now()->addHours(24));
        }
    }

    private function getExpectedColumns(string $format): array
    {
        $columns = $this->expectedColumns($format);
        if (empty($columns)) {
            throw new HeadersMissingException(
                Lang::get('validation.headers_empty_config')
            );
        }
        return $columns;
    }

    private function getExtension(string $file): string
    {
        if (!Storage::disk('local')->exists($file)) {
            throw new FileNotFoundException(Lang::get('validation.invalid_empty_file'));
        }

        $path = Storage::disk('local')->path($file);
        return pathinfo($path, PATHINFO_EXTENSION);
    }


    private function getValidator(string $extension): FeedValidationStrategy
    {
        return match ($extension) {
            'csv' => app(CsvFeedValidationStrategy::class),
            'xlsx' => app(XlsxFeedValidationStrategy::class),
            default => throw new InvalidExtensionException(
                Lang::get('validation.invalid_file_extension', [
                    'type' => $extension
                ])
            ),
        };
    }

    private function handleValidationStep(array $errors): void
    {
        $this->updateProgress();
        if (!empty($errors)) {
            $this->addErrors($errors);
        }
    }

    private function updateProgress(): void
    {
        $this->progress += self::STEP_WEIGHT;
        Cache::tags([$this->validationToken])->put('state', min(100, $this->progress), now()->addHours(24));
    }

    private function addErrors(array $errors): void
    {
        $existingErrors = Cache::tags([$this->validationToken])->get('errors', []);
        Cache::tags([$this->validationToken])->put('errors', array_merge($existingErrors, $errors), now()->addHours(24));
    }
}
