<?php

namespace Inside\CSV\Jobs;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Inside\CSV\Events\ImportFinishedEvent;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Support\Str;

class LaunchUserImport implements ShouldQueue
{
    use Queueable;
    use InteractsWithQueue;
    use Dispatchable;

    /** @var int  */
    public $timeout = 100000;

    /** @var int  */
    public $retryAfter = 180;

    /** @var int  */
    public $tries = 3;

    /** @var mixed */
    protected $user;

    /** @var string */
    protected $importedFile;

    /**
     * @param mixed $user
     * @param string $importedFile
     */
    public function __construct($user, string $importedFile)
    {
        $this->user = $user;
        $this->importedFile = $importedFile;
    }

    protected function parseOutput(string $output): void
    {
        // string should look like "[xxxx] users found"
        $usersFound = (int)Str::between(Str::before($output, 'users found'), '[', ']');

        // looks like "Total: [xxxx] users imported"
        $importedUsers = (int)Str::between(Str::after($output, 'Total:'), '[', ']');

        $disabledUsers = 0;

        if (config('csv.manual.disable', true)) {
            $disabledUsers = Str::contains($output, 'No user to disable') ? 0 : (int)Str::between(Str::after($output, 'Disabling activated users'), '[', ']');
        }

        event(new ImportFinishedEvent($usersFound, $importedUsers, $disabledUsers, $this->user));
        Log::info($usersFound . ' users have been found');
        Log::info($importedUsers . ' have been imported');
        Log::info($disabledUsers . ' have been disabled');

        // rename file after import to keep track of last import
        File::move($this->importedFile, strstr($this->importedFile, '.csv', true) . '-' . Carbon::now()->toDateTimeString() . '.csv');
    }

    public function handle(): void
    {
        $success = Artisan::call('inside:user:import', [
            'type' => 'csv',
            '--disable-not-imported' => config('csv.manual.disable', true)
        ]) === 0;

        if (!$success) {
            throw new \Exception("import failed");
        }

        $this->parseOutput(Artisan::output());
    }
}
