<?php

namespace Inside\CSV\Importers;

use Exception;
use Illuminate\Support\Arr;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use League\Csv\Reader;
use Maatwebsite\Excel\Facades\Excel;

class XLSXImporter implements ImporterInterface
{
    /** @var array */
    protected $presenters;

    public function getEntries(): array
    {
        return config('xlsx.files');
    }

    public function prepareImport(): bool
    {
        $this->presenters = config('xlsx.presenters');

        return true;
    }

    /**
     * @param array $user
     * @param array $entry
     * @param BridgeContent $bridge
     * @return array
     */
    public function getSyncAttributes($user, array $entry, BridgeContent $bridge): array // TODO ajouter typer chaque attribut une fois l'interface ImporterInterface typé
    {
        $data = [];
        foreach ($entry['fields'] as $modelField => $fileField) {
            if (is_object($fileField) && ($fileField instanceof \Closure)) {
                $data[$modelField] = $fileField($user, $bridge);
            } else {
                $data[$modelField] = $user[$fileField];
            }
            if (isset($this->presenters[$modelField]) && is_callable($this->presenters[$modelField])) {
                $data[$modelField] = $this->presenters[$modelField]($data[$modelField]);
            }
        }
        return $data;
    }

    /**
     * @param array $file
     * @param array|null $filter
     * @param array|null $test
     * @return array
     */
    public function getUsers($file, $filter, $test = null, int $limit = 0): array
    {
        if ($test) {
            return [$test];
        }

        $file = $file['path'];
        $rows = Excel::toArray((object)[], $file)[0] ?? [];
        if (empty($rows)) {
            return [];
        }

        $headers = array_map(fn ($item) => trim(strtolower($item)), $rows[0]);
        $users = array_map(fn ($row) => array_combine($headers, $row), array_slice($rows, 1));

        $postFilter = config('xlsx.postfilter');
        if ($postFilter && is_callable($postFilter)) {
            $users = $postFilter(collect($users));
        }

        return $users;
    }

    /**
     * Get the function cleanData (launched after retrieving data for a given user) from config file
     * Signature must be : function(&$data)
     *
     * @return mixed
     */
    public function getCleanData()
    {
        return config('xlsx.cleandata');
    }

    /**
     * Get the function preFlight (launched before the import) from the config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPreFlight()
    {
        return config('xlsx.preflight');
    }

    /**
     * Get the function postFlight (launched after the import) from config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPostFlight()
    {
        return config('xlsx.postflight');
    }

    public function getFilter(): ?string
    {
        return config('xlsx.filter', null);
    }

    /**
     * Get User information from AD, null if user not found.
     *
     * @param string|null $username
     * @param array $entry
     * @return array|null
     */
    public function getUserInformation($username, $entry): ?array
    {
        $users = $this->getUsers($entry, null);
        if (is_null($username)) {
            return Arr::first($users);
        }

        foreach ($users as $user) {
            if (in_array($username, $user)) {
                return $user;
            }
        }

        return null;
    }
}
